require 'rails_helper'

RSpec.describe Vods::VodsController do

  include_context 'mocked permissions'

  let(:permission) { 'vods' }

  describe 'index' do
    describe 'permissions' do
      let(:action) { :index }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          get action, params: {vod: {id: 'foo'}}
        end

        it 'redirects to the actual vod page' do
          expect(response.code).to eq('302')
        end
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          get action
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end

  end

  describe 'show' do
    describe 'permissions' do
      let(:action) { :show }
      let(:vod) { Mocks.vod }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect(Vods::Vod).to receive(:find).with(vod.id).and_return(vod)
          get action, params: {id: vod.id}
        end

        it_behaves_like 'it returns a success response'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          expect(Vods::Vod).to receive(:find).with(vod.id).and_return(vod)
          get action, params: {id: vod.id}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end

    describe 'vod appeals' do
      let(:action) { :show }
      let(:amr) { Mocks.vod_audible_magic_response }
      let(:vod) do
        vod = Mocks.vod
        vod.instance_variable_set :@audible_magic_responses, [amr]
        vod
      end

      context 'with empty track appeal' do
        include_context 'an authorized user'

        before do
          expect(Vods::Vod).to receive(:find).with(vod.id).and_return(vod)
          get action, params: {id: vod.id}
        end

        it_behaves_like 'it returns a success response'
      end
    end
  end

  describe 'destroy' do
    describe 'permissions' do
      let(:action) { :destroy }
      let(:vod) { Mocks.vod }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect(Vods::Vod).to receive(:find).with(vod.id).and_return(vod)
          expect(vod).to receive(:soft_delete).and_return(true)
          post action, params: {id: vod.id}
        end

        it_behaves_like 'it redirects the user and displays a success message'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          expect(Vods::Vod).to receive(:find).with(vod.id).and_return(vod)
          post action, params: {id: vod.id}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end

  end

  describe 'undestroy' do
    describe 'permissions' do
      let(:action) { :undestroy }
      let(:vod) { Mocks.vod }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect(Vods::Vod).to receive(:from_attributes).with(id: vod.id).and_return(vod)
          expect(vod).to receive(:undelete)
          post action, params: {id: vod.id}
        end

        it_behaves_like 'it redirects the user and displays a success message'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          expect(Vods::Vod).to receive(:from_attributes).with(id: vod.id).and_return(vod)
          post action, params: {id: vod.id}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end

  end
end
