# frozen_string_literal: true

require 'rails_helper'
require 'translate/text_translator'

RSpec.describe Tools::TextTranslator do

  let(:subject) do
    described_class.new(options: {stub_responses: true})
  end

  let(:input) do
    {from: "auto", to: "es_mx", text: "por que"}
  end

  let(:aws_mock) do
    {
      input: {
        source_language_code: "auto",
        target_language_code: "es-mx",
        text: "por que"
      },
      output: {
        successful?: true,
        translated_text: "por qué",
        source_language_code: "pt",
        target_language_code: "es-MX",
        applied_terminologies: nil,
      },
      response: double('Aws::Translate::Types::TranslateTextResponse')
    }
  end

  let(:result) do
    {
      from: {
        language_code: :pt,
        language_name: "Portuguese",
        text: "por que",
        auto_detect?: true,
        new_language?: false,
      },
      to: {
        language_code: :es_MX,
        language_name: "Spanish (Mexico)",
        text: "por qué",
      }
    }
  end

  context 'with a stubbed response' do

    before(:each) do
      # stub the aws client response
      allow(subject.instance_variable_get(:@translate_client))
        .to receive(:translate_text)
        .with(aws_mock[:input])
        .and_return(aws_mock[:response])

      # no error response
      aws_mock[:output][:error] = nil

      # let the mock return values
      allow(aws_mock[:response]).to receive_messages(aws_mock[:output])
    end

    it 'outputs translated text from input text' do
      # expect our lib to properly use the stubbed response
      expect(subject.translate(input)).to eq(result)
    end

  end

  it 'returns the language hash' do
    expect(subject.class::SUPPORTED_LANGUAGES).to be_a(Hash)
  end

end
