require 'rails_helper'
require_relative '../../service_helper'

RSpec.describe Connections::Connection do

  include ServiceHelper

  let(:user_id) {"someID"}

  %w[blizzard twitter facebook youtube league].each do |platform|
    context "#destroy" do
      it "returns true when the service call is successful" do
        allow(Connections::Base)
          .to receive(:delete).with("/admin/v2/#{platform}?user_id=#{user_id}")
          .and_return(fake_delete_response(true))
        result = Connections::Connection.destroy(user_id, platform)
        expect(result).to be_success
      end

      it "returns false when the service call is unsuccessful" do
        allow(Connections::Base)
          .to receive(:delete).with("/admin/v2/#{platform}?user_id=#{user_id}")
          .and_return(fake_delete_response(false))
        result = Connections::Connection.destroy(user_id, platform)
        expect(result).not_to be_success
      end
    end

    context "#get_all" do
      before do
        allow(Connections::Base).to receive(:get).and_return(fake_get_response(200))
      end

      it "returns false when the user does not have a connected account" do
        expect(Connections::Base)
          .to receive(:get).with("/v2/#{user_id}/#{platform}")
          .and_return(fake_get_response(404))
        result = Connections::Connection.get_all(user_id)
        expect(result[platform].has_connection).to be false
        expect(result[platform].connection_details).to be nil
      end

      it "returns true when the user has a connected account" do
        allow(Connections::Base)
          .to receive(:get)
          .and_return(fake_get_response(200, {foo: "bar"}))
        result = Connections::Connection.get_all(user_id)
        expect(result[platform].has_connection).to be true
        expect(result[platform].connection_details).to eq({foo: "bar"})
      end

      it "returns nil when the Connection service has a problem" do
        allow(Connections::Base)
          .to receive(:get)
          .and_return(fake_get_response(500))
        result = Connections::Connection.get_all(user_id)
        expect(result[platform].has_connection).to be nil
        expect(result[platform].connection_details).to be nil
      end
    end
  end
end