require 'rails_helper'

RSpec.describe Promotions::PromotionAudit do
  let(:backfill_channel) { Promotions::BackfillChannel.from_attributes(id: 'aaaaaaaa-aaaa-aaaa-aaaa-aaaabbbbccc1', title: 'title', description_raw: 'description_raw',  priority: 1, image_url: 'url', sponsored: false, active: false, user_id: 1, login: 'foo',  language: 'en', reporter: 'reporter') }

  let(:scheduled_vod) { Promotions::ScheduledVod.from_attributes(id: 'aaaaaaaa-aaaa-aaaa-aaaa-aaaabbbbccc1', vod_id: 'somemd5', title: 'title', description_raw: 'description_raw', priority: 1, image_url: 'url', sponsored: false, user_id: 1, login: 'foo',  languages: ['en'], start_date: '2017-09-18T18:00:00Z', end_date: '2017-09-19T18:00:00Z',  reporter: 'reporter') }

  context '.channel_audit_query' do
    it 'generates the query for a backfill channel' do
      expected_search = {
        resource_type: 'backfill_channel',
        resource_id: 'aaaaaaaa-aaaa-aaaa-aaaa-aaaabbbbccc1',
      }

      expect(Promotions::PromotionAudit.promotion_audit_params('backfill_channel', backfill_channel)).to eq(expected_search)
    end

    it 'generates the query for a scheduled vod' do
      expected_search = {
          resource_type: 'scheduled_vod',
          resource_id: 'aaaaaaaa-aaaa-aaaa-aaaa-aaaabbbbccc1',
      }

      expect(Promotions::PromotionAudit.promotion_audit_params('scheduled_vod', scheduled_vod)).to eq(expected_search)
    end
  end

  context '.new_channel_audit' do
    it 'generates an audit for a new channel' do
      expected_audit = History::Audit.new(
        action: 'create',
        user_type: 'ldap_user',
        user_id: 'reporter',
        resource_type: 'backfill_channel',
        resource_id: 'aaaaaaaa-aaaa-aaaa-aaaa-aaaabbbbccc1',
        changes: [
            History::ChangeSet.new(
                attribute: 'user_id',
                old_value: nil,
                new_value: 1
            ),
            History::ChangeSet.new(
                attribute: 'login',
                old_value: nil,
                new_value: 'foo'
            ),
            History::ChangeSet.new(
                attribute: 'priority',
                old_value: nil,
                new_value: 1
            ),
            History::ChangeSet.new(
                attribute: 'title',
                old_value: nil,
                new_value: 'title'
            ),
            History::ChangeSet.new(
                attribute: 'image_url',
                old_value: nil,
                new_value: 'url'
            ),
            History::ChangeSet.new(
                attribute: 'sponsored',
                old_value: nil,
                new_value: false
            ),
            History::ChangeSet.new(
                attribute: 'description_raw',
                old_value: nil,
                new_value: 'description_raw'
            ),
            History::ChangeSet.new(
                attribute: 'language',
                old_value: nil,
                new_value: 'en'
            ),
            History::ChangeSet.new(
                attribute: 'active',
                old_value: nil,
                new_value: false
            ),
        ]
      )

      expect(Promotions::PromotionAudit.new_audit('backfill_channel', backfill_channel)).to eq(expected_audit)
    end
  end

  context '.new_scheduled_vod_audit' do
    it 'generates an audit for a new scheduled vod' do
      expected_audit = History::Audit.new(
          action: 'create',
          user_type: 'ldap_user',
          user_id: 'reporter',
          resource_type: 'scheduled_vod',
          resource_id: 'aaaaaaaa-aaaa-aaaa-aaaa-aaaabbbbccc1',
          changes: [
              History::ChangeSet.new(
                  attribute: 'user_id',
                  old_value: nil,
                  new_value: 1
              ),
              History::ChangeSet.new(
                  attribute: 'login',
                  old_value: nil,
                  new_value: 'foo'
              ),
              History::ChangeSet.new(
                  attribute: 'priority',
                  old_value: nil,
                  new_value: 1
              ),
              History::ChangeSet.new(
                  attribute: 'title',
                  old_value: nil,
                  new_value: 'title'
              ),
              History::ChangeSet.new(
                  attribute: 'image_url',
                  old_value: nil,
                  new_value: 'url'
              ),
              History::ChangeSet.new(
                  attribute: 'sponsored',
                  old_value: nil,
                  new_value: false
              ),
              History::ChangeSet.new(
                  attribute: 'vod_id',
                  old_value: nil,
                  new_value: 'somemd5'
              ),
              History::ChangeSet.new(
                  attribute: 'description_raw',
                  old_value: nil,
                  new_value: 'description_raw'
              ),
              History::ChangeSet.new(
                  attribute: 'languages',
                  old_value: nil,
                  new_value: ['en']
              ),
              History::ChangeSet.new(
                  attribute: 'start_date',
                  old_value: nil,
                  new_value: '2017-09-18 18:00:00Z'
              ),
              History::ChangeSet.new(
                  attribute: 'end_date',
                  old_value: nil,
                  new_value: '2017-09-19 18:00:00Z'
              ),
          ]
      )

      expect(Promotions::PromotionAudit.new_audit('scheduled_vod', scheduled_vod)).to eq(expected_audit)
    end
  end

  context '.delete_channel_audit' do
    it 'generates an audit for a deleted channel' do
      expected_audit = History::Audit.new(
        action: 'delete',
        user_type: 'ldap_user',
        user_id: 'reporter',
        resource_type: 'backfill_channel',
        resource_id: 'aaaaaaaa-aaaa-aaaa-aaaa-aaaabbbbccc1',
        changes: [
            History::ChangeSet.new(
                attribute: 'user_id',
                old_value: 1,
                new_value: nil
            ),
            History::ChangeSet.new(
                attribute: 'login',
                old_value: 'foo',
                new_value: nil
            ),
            History::ChangeSet.new(
                attribute: 'priority',
                old_value: 1,
                new_value: nil
            ),
            History::ChangeSet.new(
                attribute: 'title',
                old_value: 'title',
                new_value: nil
            ),
            History::ChangeSet.new(
                attribute: 'image_url',
                old_value: 'url',
                new_value: nil
            ),
            History::ChangeSet.new(
                attribute: 'sponsored',
                old_value: false,
                new_value: nil
            ),
            History::ChangeSet.new(
                attribute: 'description_raw',
                old_value: 'description_raw',
                new_value: nil
            ),
            History::ChangeSet.new(
                attribute: 'language',
                old_value: 'en',
                new_value: nil
            ),
            History::ChangeSet.new(
                attribute: 'active',
                old_value: false,
                new_value: nil
            ),
        ]
      )

      expect(Promotions::PromotionAudit.delete_audit('backfill_channel', backfill_channel)).to eq(expected_audit)
    end
  end

  context '.delete_scheduled_vod_audit' do
    it 'generates an audit for a deleted scheduled vod' do
      expected_audit = History::Audit.new(
          action: 'delete',
          user_type: 'ldap_user',
          user_id: 'reporter',
          resource_type: 'scheduled_vod',
          resource_id: 'aaaaaaaa-aaaa-aaaa-aaaa-aaaabbbbccc1',
          changes: [
              History::ChangeSet.new(
                  attribute: 'user_id',
                  old_value: 1,
                  new_value: nil
              ),
              History::ChangeSet.new(
                  attribute: 'login',
                  old_value: 'foo',
                  new_value: nil
              ),
              History::ChangeSet.new(
                  attribute: 'priority',
                  old_value: 1,
                  new_value: nil
              ),
              History::ChangeSet.new(
                  attribute: 'title',
                  old_value: 'title',
                  new_value: nil
              ),
              History::ChangeSet.new(
                  attribute: 'image_url',
                  old_value: 'url',
                  new_value: nil
              ),
              History::ChangeSet.new(
                  attribute: 'sponsored',
                  old_value: false,
                  new_value: nil
              ),
              History::ChangeSet.new(
                  attribute: 'vod_id',
                  old_value: 'somemd5',
                  new_value: nil
              ),
              History::ChangeSet.new(
                  attribute: 'description_raw',
                  old_value: 'description_raw',
                  new_value: nil
              ),
              History::ChangeSet.new(
                  attribute: 'languages',
                  old_value: ['en'],
                  new_value: nil
              ),
              History::ChangeSet.new(
                  attribute: 'start_date',
                  old_value: '2017-09-18 18:00:00Z',
                  new_value: nil
              ),
              History::ChangeSet.new(
                  attribute: 'end_date',
                  old_value: '2017-09-19 18:00:00Z',
                  new_value: nil
              ),
          ]
      )

      expect(Promotions::PromotionAudit.delete_audit('scheduled_vod', scheduled_vod)).to eq(expected_audit)
    end
  end

  context '.update_channel_audit' do
    it 'generates an audit for a updated channel' do
      expect(backfill_channel).to receive(:changes).and_return({
        "title" => {old_value: 'title', new_value: 'New title'},
        "priority" => {old_value: 1, new_value: 10},
        "sponsored" => {old_value: false, new_value: true}
      })

      expected_audit = History::Audit.new(
        action: 'update',
        user_type: 'ldap_user',
        user_id: 'reporter',
        resource_type: 'backfill_channel',
        resource_id: 'aaaaaaaa-aaaa-aaaa-aaaa-aaaabbbbccc1',
        changes: [
          History::ChangeSet.new(
            attribute: 'title',
            old_value: 'title',
            new_value: 'New title'
          ),
          History::ChangeSet.new(
            attribute: 'priority',
            old_value: 1,
            new_value: 10
          ),
          History::ChangeSet.new(
            attribute: 'sponsored',
            old_value: false,
            new_value: true
          )
        ]
      )

      expect(Promotions::PromotionAudit.update_audit('backfill_channel', backfill_channel)).to eq(expected_audit)
    end
  end

  context '.update_scheduled_vod_audit' do
    it 'generates an audit for a updated scheduled vod' do
      expect(scheduled_vod).to receive(:changes).and_return({
        "title" => {old_value: 'title', new_value: 'New title'},
        "priority" => {old_value: 1, new_value: 10},
        "sponsored" => {old_value: false, new_value: true}
      })

      expected_audit = History::Audit.new(
        action: 'update',
        user_type: 'ldap_user',
        user_id: 'reporter',
        resource_type: 'scheduled_vod',
        resource_id: 'aaaaaaaa-aaaa-aaaa-aaaa-aaaabbbbccc1',
        changes: [
          History::ChangeSet.new(
            attribute: 'title',
            old_value: 'title',
            new_value: 'New title'
          ),
          History::ChangeSet.new(
            attribute: 'priority',
            old_value: 1,
            new_value: 10
          ),
          History::ChangeSet.new(
            attribute: 'sponsored',
            old_value: false,
            new_value: true
          )
        ]
      )

      expect(Promotions::PromotionAudit.update_audit('scheduled_vod', scheduled_vod)).to eq(expected_audit)
    end
  end
end
