# frozen_string_literal: true

require 'rails_helper'

module Staff
  describe Member do
    let(:user_id) { 'my-user-id' }

    describe 'find' do
      before(:each) do
        expect(Base).to receive(:get)
          .with("/ldap/users/#{user_id}")
          .and_return(response)
      end

      context 'when searching for a valid user' do
        let(:user_attributes) do
          {
          'cn' => 'my-user-cn',
          'email' => 'my-user-email',
          'groups' => 'my-user-groups'
        }
        end

        let(:user) do
          Member.new(
            id: user_id,
            cn: user_attributes['cn'],
            email: user_attributes['email'],
            groups: user_attributes['groups']
          )
        end

        let(:response) do
          Faraday::Response.new.tap do |response|
            allow(response).to receive(:success?).and_return(true)
            allow(response).to receive(:body).and_return({
              'data' => { 'attributes' => user_attributes }
            })
          end
        end

        it 'returns successfully' do
          response = Member.find(user_id)
          user.attributes.each do |k, v|
            expect(response.attributes[k]).to equal(v)
          end
        end
      end

      context 'when getting an error' do
        let(:error_detail) { 'my-error-detail' }
        let(:response) do
          Faraday::Response.new.tap do |response|
            allow(response).to receive(:success?).and_return(false)
            allow(response).to receive(:body).and_return({
              'errors' => [{ 'detail' => error_detail }]
            })
          end
        end

        it 'returns sets errors on the user' do
          response = Member.find(user_id)
          expect(response.errors.empty?).to be(false)
          expect(response.id).to eq(user_id)
        end
      end
    end

    describe 'activity_logs' do
      let(:client) { double('Twitch::ActivityLog::Client') }
      let(:user) { Member.new }
      let(:response) { { my: 'response' } }

      before(:each) do
        allow(Twitch::ActivityLog::Client).to receive(:new).and_return(client)
        allow(client).to receive(:query).and_return(response)
      end

      it 'runs the query' do
        activity_logs = user.activity_logs()
        expect(activity_logs).to eq(response)
      end
    end
  end
end
