require 'rails_helper'

module Zuma
  describe UserBlock do
    describe "#all" do
      describe "with no id" do
        it "returns an empty array" do
          expect(UserBlock.all(nil)).to eq([])
        end
      end

      describe "with a valid id and blocked users" do
        let(:id) { 1234 }
        let(:blocked_user_ids) { ["345", "678"] }

        request_body = '{"user_id":"1234"}'

        it "returns an array of blocked user IDs" do
          resp = double(:response,
            {
              body: {
                "blocked_user_ids" => ["345", "678"]
              },
              success?: true
            })
          allow(Base).to receive(:post).with("/v1/users/blocks/get", {body: request_body}).and_return(resp)
          user_block = UserBlock.all(id)

          expect(user_block.blocked_user_ids).to eq(blocked_user_ids)
        end
      end

      describe "with a valid id and no blocked users" do
        let(:id) { 1234 }
        let(:blocked_user_ids) { [] }

        request_body = '{"user_id":"1234"}'

        it "returns an array of blocked user IDs" do
          resp = double(:response,
            {
              body: {
                "blocked_user_ids" => nil
              },
              success?: true
            })
          allow(Base).to receive(:post).with("/v1/users/blocks/get", {body: request_body}).and_return(resp)
          user_block = UserBlock.all(id)

          expect(user_block).to eq(blocked_user_ids)
        end
      end

      describe "and the underlying service errors" do
        let(:id) { 1234 }
        let(:blocked_user_ids) { [] }

        request_body = '{"user_id":"1234"}'

        it "returns an array of blocked user IDs" do
          resp = double(:response,
            {
              success?: false
            })
          allow(Base).to receive(:post).with("/v1/users/blocks/get", {body: request_body}).and_return(resp)
          user_block = UserBlock.all(id)

          expect(user_block).to eq(blocked_user_ids)
        end
      end

    end

    describe "#find" do
      describe "with valid ids" do
        let(:id) { 1234 }
        let(:other_id) { 5678 }

        request_body = '{"user_id":"1234","blocked_user_id":"5678"}'

        it "returns whether the user is blocking the other" do
          resp = double(:response,
            {
              body: {
                "user_id" => "1234",
                "blocked_user_id" => "5678",
                "is_blocked" => true
              },
              success?: true
            })
          allow(Base).to receive(:post).with("/v1/users/blocks/is_blocked", {body: request_body}).and_return(resp)
          user_block = UserBlock.find(id, other_id)

          expect(user_block.is_blocked).to eq(true)
        end
      end
    end
  end
end
