# ECS Container role
resource "aws_iam_role" "app" {
  name = "admin-panel-${var.environment}"

  assume_role_policy = "${data.aws_iam_policy_document.app-arp.json}"
}

data "aws_iam_policy_document" "app-arp" {
  statement {
    actions = [
      "sts:AssumeRole",
    ]

    principals {
      type        = "Service"
      identifiers = ["ecs-tasks.amazonaws.com"]
    }

    principals {
      type        = "AWS"
      identifiers = ["${local.root_arn}"]
    }
  }
}

resource "aws_iam_role_policy" "app" {
  role   = "${aws_iam_role.app.id}"
  policy = "${data.aws_iam_policy_document.app.json}"
}

data "aws_iam_policy_document" "app" {
  statement {
    actions = [
      "sts:AssumeRole",
    ]

    resources = [
      "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/admin-panel-${var.environment == "canary" ? "production" : var.environment}",
      "arn:aws:iam::180116294062:role/malachai/*",
      "${var.service_role_arns}",
    ]
  }

  statement {
    actions = [
      "sts:AssumeRole",
    ]

    resources = [
      ${var.environment == "staging" ? "arn:aws:iam::259766472440:role/sam-dev-harrding-TwitchVa-AdminPanelInvokeLambdaRo-1NQEWHBWGI2DO" : "arn:aws:iam::751728601281:role/TwitchVaderLambda-AdminPanelInvokeLambdaRole-12M62GOAJH78O"},
    ]
  }

  statement {
    actions   = ["sns:publish"]
    resources = ["${var.pushy_sns_topic}"]
  }

  statement {
    // See https://wiki.twitch.com/display/SSE/S2S2+Service+Onboarding#S2S2ServiceOnboarding-EnsureIAMPermissions
    actions   = ["execute-api:Invoke"]
    resources = ["arn:aws:execute-api:*:985585625942:*"]
  }
  statement {
    actions = [
      "sts:AssumeRole",
    ]

    resources = [
      "arn:aws:iam::${var.environment == "staging" ? "589506732331" : "316550374861"}:role/fraud_report_lambda_role",
    ]
  }
}

resource "aws_iam_role_policy_attachment" "app" {
  role       = "${aws_iam_role.execution.name}"
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonECSTaskExecutionRolePolicy"
}

resource "aws_iam_role_policy_attachment" "s3" {
  role       = "${aws_iam_role.app.name}"
  policy_arn = "${aws_iam_policy.s3.arn}"
}

resource "aws_iam_role_policy_attachment" "TranslateReadOnly" {
  role       = "${aws_iam_role.app.name}"
  policy_arn = "arn:aws:iam::aws:policy/TranslateReadOnly"
}

# S3 image upload role
resource "aws_iam_policy" "s3" {
  name   = "admin-panel-${var.environment}-upload-image"
  policy = "${data.aws_iam_policy_document.s3.json}"
}

data "aws_iam_policy_document" "s3" {
  statement {
    actions = [
      "s3:Put*",
      "s3:ListBucket",
      "s3:GetObject",  # required to check for file existence before 
    ]

    resources = [
      "arn:aws:s3:::ttv-user-pictures-prod/hosted_images/*",
      "arn:aws:s3:::twitch-admin-panel-image-upload-${var.environment}/hosted_images/*",
      "arn:aws:s3:::twitch-admin-panel-image-upload-${var.environment}",
      "arn:aws:s3:::ttv-user-pictures-prod",
    ]
  }
}

# Execution IAM role for ECS Container
resource "aws_iam_role" "execution" {
  name               = "admin-panel-ecs-exec-${var.environment}"
  assume_role_policy = "${data.aws_iam_policy_document.execution.json}"
}

data "aws_iam_policy_document" "execution" {
  statement {
    actions = [
      "sts:AssumeRole",
    ]

    principals {
      type        = "Service"
      identifiers = ["ecs-tasks.amazonaws.com", "ecs.amazonaws.com"]
    }
  }
}

# Deploy to ECR IAM
resource "aws_iam_role" "deploy" {
  name               = "admin-panel-deploy-${var.environment}"
  assume_role_policy = "${data.aws_iam_policy_document.deploy-arp.json}"
}

data "aws_iam_policy_document" "deploy-arp" {
  statement = {
    actions = [
      "sts:AssumeRole",
    ]

    principals {
      type        = "AWS"
      identifiers = ["${var.allowed_deployment_arns}"]
    }
  }
}

resource "aws_iam_role_policy" "deploy" {
  role   = "${aws_iam_role.deploy.id}"
  policy = "${data.aws_iam_policy_document.deploy-p.json}"
}

data "aws_iam_policy_document" "deploy-p" {
  statement {
    actions = [
      "ecs:RegisterTaskDefinition",
      "ecs:DescribeServices",       # for deployment checks
      "ecs:UpdateService",
      "ecr:DescribeRepositories",
      "ecr:GetAuthorizationToken",
    ]

    resources = [
      "*",
    ]
  }

  statement {
    actions = [
      "ecr:*",
    ]

    resources = [
      "${aws_ecr_repository.app.arn}",
    ]
  }

  statement {
    actions = [
      "iam:PassRole",
    ]

    resources = [
      "${aws_iam_role.app.arn}",
      "${aws_iam_role.execution.arn}",
    ]
  }
}
