resource "aws_alb" "app" {
  name     = "admin-panel-${var.environment}"
  internal = true

  security_groups = [
    "${var.twitch_subnet_sg}",
  ]

  subnets = "${var.twitch_subnet}"

  access_logs {
    bucket  = "${aws_s3_bucket.app-alb-logs.bucket}"
    prefix  = "admin-panel-alb-logs"
    enabled = true
  }

  tags {
    Service     = "admin-panel"
    Environment = "${var.environment}"
  }

  depends_on = [
    "aws_s3_bucket_policy.app-alb-logs",
  ]
}

resource "aws_alb_target_group" "app" {
  name        = "admin-panel-${var.environment}"
  port        = "8000"
  protocol    = "HTTP"
  vpc_id      = "${var.vpc_id}"
  target_type = "ip"

  health_check {
    matcher = "200"
    path    = "/elb-health"
  }
}

resource "aws_alb_listener" "app" {
  load_balancer_arn = "${aws_alb.app.arn}"
  port              = "443"
  protocol          = "HTTPS"
  ssl_policy        = "ELBSecurityPolicy-2015-05"
  certificate_arn   = "${aws_acm_certificate.app.arn}"

  default_action {
    target_group_arn = "${aws_alb_target_group.app.arn}"
    type             = "forward"
  }
}

resource "aws_alb_target_group" "http" {
  name        = "admin-panel-http-${var.environment}"
  port        = "80"
  protocol    = "HTTP"
  vpc_id      = "${var.vpc_id}"
  target_type = "ip"

  health_check {
    matcher = "200"
    path    = "/health"
  }
}

resource "aws_alb_listener" "app-http" {
  load_balancer_arn = "${aws_alb.app.arn}"
  port              = "80"
  protocol          = "HTTP"

  default_action {
    target_group_arn = "${aws_alb_target_group.http.arn}"
    type             = "forward"
  }
}

resource "aws_s3_bucket" "app-alb-logs" {
  bucket = "admin-panel-${var.environment}-app-alb-logs"
  acl    = "private"

  tags {
    Name        = "admin-panel-alb-logs"
    Environment = "${var.environment}"
  }
}

resource "aws_s3_bucket_policy" "app-alb-logs" {
  bucket = "${aws_s3_bucket.app-alb-logs.bucket}"
  policy = "${data.aws_iam_policy_document.app-alb-logs-bp.json}"
}

data "aws_iam_policy_document" "app-alb-logs-bp" {
  statement {
    actions = [
      "s3:PutObject",
    ]

    resources = [
      "${aws_s3_bucket.app-alb-logs.arn}/*",
    ]

    principals = {
      type        = "AWS"
      identifiers = ["${var.alb_account_id}"]
    }
  }
}
