provider "aws" {} # proxy provider to consume incoming providers implicitly

resource "aws_s3_bucket" "image_upload" {
  bucket = "twitch-admin-panel-image-upload-${var.env}"
  acl    = "private"                                    # bucket owner has full control, nobody else gets anything

  cors_rule {
    allowed_headers = ["*"]
    allowed_methods = ["PUT"]                        # only uploads are allowed from browsers
    allowed_origins = ["${var.cors_allowed_origin}"]
    expose_headers  = ["ETag"]
    max_age_seconds = 3000
  }

  tags {
    Owner                       = "admin-services@twitch.tv"
    Name                        = "admin-panel-image-upload"
    Environment                 = "${var.env}"
    Provisioner                 = "terraform"
    "twitch:s3:purpose"         = "hosted-images"
    "twitch:catalog:service:id" = "5"                        # catalog.xarth.tv
  }

  lifecycle_rule {
    enabled = "${var.lifecycle_rule_enabled}"
    prefix  = "${var.lifecycle_rule_prefix}"

    expiration {
      days = "${var.lifecycle_rule_expiration_days}"
    }
  }
}

resource "aws_s3_bucket_public_access_block" "block_public_policy" {
  bucket = "${aws_s3_bucket.image_upload.id}"

  /*
  block_public_acls       : Disallows the use of *PUBLIC* ACLs for future bucket or object policy updates
  ignore_public_acls      : Ignores any existing *PUBLIC* ACLs when evaluating access to a bucket or object
  restrict_public_buckets : Restricts access of buckets and objects currently classified as *PUBLIC* to the Account Owner and AWS Services
  block_public_policy     : Disallows PUT of bucket policies that would classify a directory path in the bucket as *PUBLIC* / Public Access / World Readable
  */

  block_public_acls       = false
  ignore_public_acls      = false
  restrict_public_buckets = false
  block_public_policy     = true
  depends_on              = ["aws_s3_bucket.image_upload"]
}

data "aws_iam_policy_document" "s3_reader_uploader_access" {
  statement {
    sid = "HostedImagesUploaderAccess"

    effect = "Allow"

    principals {
      type        = "AWS"
      identifiers = ["${var.upload_service_role}"] # only requests or urls signed by this role can upload
    }

    actions = [
      "s3:Put*",
    ]

    resources = [
      "${aws_s3_bucket.image_upload.arn}/hosted_images/*",
    ]
  }

}

resource "aws_s3_bucket_policy" "reader_uploader_access" {
  bucket     = "${aws_s3_bucket.image_upload.id}"
  policy     = "${data.aws_iam_policy_document.s3_reader_uploader_access.json}"
  depends_on = ["aws_s3_bucket_public_access_block.block_public_policy"]
}
