// Package iprange provides known IP ranges for the admin AWS accounts.
// Sources include:
//   - AWS IP ranges: https://docs.aws.amazon.com/general/latest/gr/aws-ip-ranges.html
//   - DUO IP ranges: https://help.duo.com/s/article/1337?language=en_US
//   - Rollbar IP Addresses: https://docs.rollbar.com/docs/rollbar-ip-addresses
//   - Twitch IP Ranges: Mostly taken from SYS configured security groups. See
//     the implementation of GetTwitchIPRanges for precise rules.
package iprange

import (
	"context"
	"net"
)

// IPRange represents an ip range
type IPRange interface {
	Contains(net.IP) bool
	Description() string
}

// GetIPRanges returns all known ip ranges
func GetIPRanges(ctx context.Context) ([]IPRange, error) {
	var err error
	var ranges []IPRange

	ranges = append(ranges, GetRollbarIPRanges()...)

	twitch, err := GetTwitchIPRanges()
	if err != nil {
		return nil, err
	}
	ranges = append(ranges, twitch...)

	duo, err := GetDuoIPRanges()
	if err != nil {
		return nil, err
	}
	ranges = append(ranges, duo...)

	aws, err := GetAWSIPRanges(ctx)
	if err != nil {
		return nil, err
	}
	for _, r := range aws {
		ranges = append(ranges, r)
	}

	dns, err := GetDNSIPRanges()
	if err != nil {
		return nil, err
	}
	ranges = append(ranges, dns...)

	return ranges, nil
}
