package c7

import (
	"testing"

	"bytes"
	"code.justin.tv/amzn/C7-go/internal/rule"
	"fmt"
	"github.com/stretchr/testify/assert"
	"time"
)

func GetConfig(t *testing.T, strrule string, selectors ...string) (*C7, *C7Set) {
	t.Helper()
	set, err := Parse(strrule)
	assert.NoError(t, err)
	appConfig := NewC7(*set, selectors...)
	return appConfig, set
}

func singleStringC7() *C7 {
	s := "s"
	rules := rule.Rules{
		{
			Selectors: []string{"*"},
			Value:     rule.C7Value{StrValue: &s},
			Keys:      []string{"k"},
			Namespace: "ns",
		},
	}
	return NewC7(C7Set{rules: rules}, "a")
}

func TestC7GetHasValue(t *testing.T) {
	ac := singleStringC7()
	assert.True(t, ac.HasValue("ns", "k"))
}

func TestC7GetString(t *testing.T) {
	ac := singleStringC7()
	s, err := ac.string("ns", "k")
	assert.NoError(t, err)
	assert.Equal(t, "s", *s)
}

func TestC7GetStringIntoBool(t *testing.T) {
	ac := singleStringC7()
	b, err := ac.bool("ns", "k")
	assert.Error(t, err)
	assert.Equal(t, "failed to get ns:k, want: bool, have: string", err.Error())
	assert.Nil(t, b)
}

func TestC7GetStringIntoInt(t *testing.T) {
	ac := singleStringC7()
	i, err := ac.int64("ns", "k")
	assert.Error(t, err)
	assert.Equal(t, "failed to get ns:k, want: int64, have: string", err.Error())
	assert.Nil(t, i)
}

func singleIntC7() *C7 {
	i := int64(32)
	rules := rule.Rules{
		{
			Selectors: []string{"*"},
			Value:     rule.C7Value{IntValue: &i},
			Keys:      []string{"k"},
			Namespace: "ns",
		},
	}
	return NewC7(C7Set{rules: rules}, "a")
}

func TestC7GetInt(t *testing.T) {
	ac := singleIntC7()
	i, err := ac.int64("ns", "k")
	assert.NoError(t, err)
	assert.Equal(t, int64(32), *i)
}

func TestC7GetIntIntoString(t *testing.T) {
	ac := singleIntC7()
	s, err := ac.string("ns", "k")
	assert.Error(t, err)
	assert.Equal(t, "failed to get ns:k, want: string, have: int64", err.Error())
	assert.Nil(t, s)
}

func TestC7GetBool(t *testing.T) {
	b := true
	rules := rule.Rules{
		{
			Selectors: []string{"*"},
			Value:     rule.C7Value{BoolValue: &b},
			Keys:      []string{"k"},
			Namespace: "ns",
		},
	}
	ac := NewC7(C7Set{rules: rules}, "a")
	actual, err := ac.bool("ns", "k")
	assert.NoError(t, err)
	assert.Equal(t, true, *actual)
}

func TestFillIntFails(t *testing.T) {
	var i int64
	configText := "*:NS:KEY = 32;\n"

	appConfig, _ := GetConfig(t, configText, "a")

	err := appConfig.FillWithNamespace("NS", i)
	assert.Error(t, err)
	assert.Equal(t, `c7 can only fill pointers, passed "int64"`, err.Error())
}

func TestFillIntPtrFails(t *testing.T) {
	var i int64
	configText := "*:NS:KEY = 32;\n"

	appConfig, _ := GetConfig(t, configText, "a")

	err := appConfig.FillWithNamespace("NS", &i)
	assert.Error(t, err)
	assert.Equal(t, `c7 can only fill structs, passed "int64"`, err.Error())
}

func TestFillNonPtrStructFails(t *testing.T) {
	type C struct {
		S string
	}
	configText := "*:NS:KEY = 32;\n"

	appConfig, _ := GetConfig(t, configText, "a")

	err := appConfig.FillWithNamespace("NS", C{})
	assert.Error(t, err)
	assert.Equal(t, `c7 can only fill pointers, passed "struct"`, err.Error())
}

func TestFillStringKey(t *testing.T) {
	type Config struct {
		Value string `c7:"KEY"`
	}

	type BadConfigNoKey struct {
		Value string `c7:""`
	}

	type ConfigWithNamespaceNotYetSupported struct {
		Value string `c7:"NS:KEY"`
	}

	type ConfigWithWrongType struct {
		Value bool `c7:"KEY"`
	}

	type ConfigWithTwoKeys struct {
		Value bool `c7:"k1.k2"`
	}

	configTxt := "*:NS:KEY = \"hi\";\n"
	appConfig, _ := GetConfig(t, configTxt, "a")

	config := &Config{}
	ns := "NS"
	err := appConfig.FillWithNamespace(ns, config)
	assert.NoError(t, err)
	assert.Equal(t, "hi", config.Value)

	badConfig := &BadConfigNoKey{}
	err = appConfig.FillWithNamespace(ns, badConfig)
	assert.Error(t, err)
	assert.Equal(t, "struct tag must provide a key, tag: \"\", field: BadConfigNoKey.Value", err.Error())

	cwnsnys := &ConfigWithNamespaceNotYetSupported{}
	err = appConfig.FillWithNamespace(ns, cwnsnys)
	assert.Error(t, err)

	cwtk := &ConfigWithTwoKeys{}
	err = appConfig.FillWithNamespace(ns, cwtk)
	assert.Error(t, err)
	assert.Equal(t, `invalid keys: "k1.k2"`, err.Error())

	cwt := &ConfigWithWrongType{}
	err = appConfig.FillWithNamespace(ns, cwt)
	assert.Error(t, err)
	assert.Equal(t, "failed to fill ConfigWithWrongType.Value: failed to get NS:KEY, want: bool, have: string", err.Error())
}

func TestFillStringUnexportedField(t *testing.T) {
	type Config struct {
		value string `c7:"KEY"`
	}
	configTxt := "*:NS:KEY = \"hi\";\n"
	appConfig, _ := GetConfig(t, configTxt, "a")
	config := &Config{}
	err := appConfig.FillWithNamespace("NS", config)
	assert.Error(t, err)
	assert.Equal(t, "field c7.Config.value can't be set", err.Error())
	assert.Equal(t, "", config.value)
}

func stringIt(t *testing.T, appSet *C7Set) string {
	buf := bytes.Buffer{}
	err := appSet.Write(&buf)
	assert.NoError(t, err)
	return string(buf.Bytes())
}

func TestFillBool(t *testing.T) {
	type Config struct {
		Value bool `c7:"KEY"`
	}

	configTxt := "*:NS:KEY = true;\na:NS:KEY = false;\n"
	appConfig, appSet := GetConfig(t, configTxt, "a")

	config := &Config{}
	err := appConfig.FillWithNamespace("NS", config)
	assert.NoError(t, err)
	assert.False(t, config.Value)

	s := stringIt(t, appSet)
	assert.Equal(t, s, configTxt)
}

func TestFillInt(t *testing.T) {
	type Config struct {
		Value int64 `c7:"KEY"`
	}

	configTxt := "*.*.*:NS:KEY = 30;\n"
	appConfig, appSet := GetConfig(t, configTxt, "a", "b", "c")

	config := &Config{}
	err := appConfig.FillWithNamespace("NS", config)
	assert.NoError(t, err)
	assert.Equal(t, config.Value, int64(30))

	s := stringIt(t, appSet)
	assert.Equal(t, s, configTxt)
}

func TestFillErrorInt(t *testing.T) {
	type Config struct {
		Value int `c7:"KEY"`
	}
	configTxt := "*.*.*:NS:KEY = 30;\n"
	appConfig, _ := GetConfig(t, configTxt, "a", "b", "c")

	config := &Config{}
	err := appConfig.FillWithNamespace("NS", config)
	assert.Error(t, err)
	assert.Equal(t, "attempted to set Config.Value for unsupported field type: int", err.Error())
}

func TestFillDuration(t *testing.T) {
	type Config struct {
		Duration time.Duration `c7:"KEY"`
	}

	// milliseconds
	appConfig, _ := GetConfig(t, "*.*:NS:KEY = 300s;\n", "a", "b")

	config := &Config{}
	err := appConfig.FillWithNamespace("NS", config)
	assert.NoError(t, err)
	assert.Equal(t, float64(300), config.Duration.Seconds())
}

func TestFillErrorDuation(t *testing.T) {
	type Config struct {
		Value time.Duration `c7:"KEY"`
	}
	configTxt := "*.*.*:NS:KEY = \"hi\";\n"
	appConfig, _ := GetConfig(t, configTxt, "a", "b", "c")

	config := &Config{}
	err := appConfig.FillWithNamespace("NS", config)
	assert.Error(t, err)
	assert.Equal(t, "failed to fill Config.Value: failed to get NS:KEY, want: time.Duration, have: string", err.Error())
}

func TestDefaultAndSpefificTwoKeyDev(t *testing.T) {
	type Config struct {
		Value bool `c7:"Noop"`
	}
	appConfig, _ := GetConfig(t, "*.*:Statsd:Noop=true;\n*.development:Statsd:Noop=false;\n", "us-west-2", "development")
	config := &Config{}
	err := appConfig.FillWithNamespace("Statsd", config)
	assert.NoError(t, err)
	assert.False(t, config.Value)
}

func TestDefaultAndSpecificTwoKeyProd(t *testing.T) {
	type Config struct {
		Value bool `c7:"Noop"`
	}
	appConfig, _ := GetConfig(t, "*.*:Statsd:Noop=true;\n*.development:Statsd:Noop=false;\n", "us-west-2", "production")
	config := &Config{}
	err := appConfig.FillWithNamespace("Statsd", config)
	assert.NoError(t, err)
	assert.True(t, config.Value)
}

func TestWriteConfigToDisk(t *testing.T) {
	strCfg := "*.*:Statsd:Noop = true;\n*.development:Statsd:Noop = false;\n"
	_, set := GetConfig(t, strCfg, "us-west-2", "production")
	buf := bytes.Buffer{}
	err := set.Write(&buf)
	assert.NoError(t, err)
	assert.Equal(t, strCfg, string(buf.Bytes()))
}

func BenchmarkFillWithNamespace(b *testing.B) {
	type Config struct {
		b bool   `c7:"b"`
		i int64  `c7:"i"`
		s string `c7:"s"`
	}

	nSel := 4
	nRules := 2048
	nRules = nSel * 6 * nRules
	rules := make(rule.Rules, 0, nRules)

	in := int64(3)
	st := "s"
	bl := false
	selectors := make([]string, nSel, nSel)
	stripeSelectors := make([]string, nSel, nSel)
	for i := range selectors {
		selectors[i] = "*"
		stripeSelectors[i] = "*"
	}

	for i := 0; i < nRules/nSel/6; i++ {
		for s := 0; s < nSel; s++ {
			selectors[s] = "a"
			tmpSelectors := make([]string, len(selectors))
			copy(tmpSelectors, selectors)
			rules = append(rules, &rule.Rule{
				Selectors: tmpSelectors,
				Value:     rule.C7Value{IntValue: &in},
				Keys:      []string{"b"},
				Namespace: "ns",
			})

			rules = append(rules, &rule.Rule{
				Selectors: tmpSelectors,
				Value:     rule.C7Value{BoolValue: &bl},
				Keys:      []string{"b"},
				Namespace: "ns",
			})

			rules = append(rules, &rule.Rule{
				Selectors: tmpSelectors,
				Value:     rule.C7Value{StrValue: &st},
				Keys:      []string{"b"},
				Namespace: "ns",
			})

			tmpSelectors = make([]string, len(selectors))
			copy(stripeSelectors, selectors)
			tmpSelectors[s] = "a"

			rules = append(rules, &rule.Rule{
				Selectors: tmpSelectors,
				Value:     rule.C7Value{IntValue: &in},
				Keys:      []string{"b"},
				Namespace: "ns",
			})

			rules = append(rules, &rule.Rule{
				Selectors: tmpSelectors,
				Value:     rule.C7Value{BoolValue: &bl},
				Keys:      []string{"b"},
				Namespace: "ns",
			})

			rules = append(rules, &rule.Rule{
				Selectors: tmpSelectors,
				Value:     rule.C7Value{StrValue: &st},
				Keys:      []string{"b"},
				Namespace: "ns",
			})
		}
	}

	cfg := NewC7(C7Set{rules: rules}, selectors...)
	c := &Config{}

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		cfg.FillWithNamespace("ns", &c)
	}
}

func TestMergeEmpty(t *testing.T) {
	s1, err := Parse("")
	assert.NoError(t, err)

	s2, err := Parse("")
	assert.NoError(t, err)

	err = s1.Merge(s2)
	assert.NoError(t, err)

	assert.Equal(t, 0, len(s1.rules))
}

func TestMergeIntRulesDifferentValues(t *testing.T) {
	s1, err := Parse("s1.s2:n:k = 35;")
	assert.NoError(t, err)

	s2, err := Parse("s1.s2:n:k = 42;")
	assert.NoError(t, err)

	err = s1.Merge(s2)
	assert.NoError(t, err)

	assert.Equal(t, 1, len(s1.rules))
	assert.Equal(t, int64(42), *s1.rules[0].Value.IntValue)
}

func TestMergingStringInt(t *testing.T) {
	s1, err := Parse("s1.s2:n:k = 35;")
	assert.NoError(t, err)

	s2, err := Parse(`s1.s2:n:k = "hello";`)
	assert.NoError(t, err)

	err = s1.Merge(s2)
	assert.Error(t, err)
	assert.Equal(t, err.Error(), "Trying to replace s1.s2:n:k type int64 with type string")
	assert.Equal(t, int64(35), *s1.rules[0].Value.IntValue)
}

func TestMergingWithWildcard(t *testing.T) {
	s1, err := Parse("s1.s2:n:k = 35;")
	assert.NoError(t, err)

	s2, err := Parse("s1.*:n:k = 42;")
	assert.NoError(t, err)

	err = s1.Merge(s2)
	assert.NoError(t, err)

	assert.Equal(t, 2, len(s1.rules))
}

func BenchmarkAggregator(b *testing.B) {
	bench := func(n int) {
		b.Run(fmt.Sprintf("%d rules", n), func(b *testing.B) {
			rules := make([]*rule.Rule, n)
			v := new(int64)
			*v = 1
			for i := 0; i < n; i++ {
				rules[i] = &rule.Rule{
					Selectors: []string{"s1", "s2", "s3", "s4", "s5"},
					Namespace: "namespace",
					Value:     rule.C7Value{IntValue: v},
					Keys:      []string{"k1", "k2", "k3", "k4", "k5", fmt.Sprintf("k%d", i)},
				}
			}

			set := NewC7Set()
			b.ResetTimer()

			for j := 0; j < n; j++ {
				for i := 0; i < b.N; i++ {
					s2 := NewC7Set()
					s2.addRule(rules[j])
					_ = set.Merge(s2)
				}
			}
		})
	}

	bench(10)
	bench(100)
	bench(1000)
	bench(10000)
}
