package ast

import (
	"testing"

	"code.justin.tv/amzn/C7-go/internal/rule"
	"github.com/stretchr/testify/assert"
)

func assertLoadConfig(t *testing.T, strRules string) rule.Rules {
	t.Helper()
	rules, err := Parse(strRules)
	assert.NoError(t, err)
	assert.NotNil(t, strRules)
	return rules
}

func TestBroken(t *testing.T) {
	rules, err := Parse("*.*.*:a,b")
	assert.Error(t, err)
	assert.Nil(t, rules)
	assert.Empty(t, rules)
}

func TestEmptyConfig(t *testing.T) {
	rules, err := Parse("")
	assert.NoError(t, err)
	assert.NotNil(t, rules)
	assert.Empty(t, rules)
}

func TestCommentOnlyConfig(t *testing.T) {
	rules, err := Parse("// HI THERE")
	assert.NoError(t, err)
	assert.NotNil(t, rules)
	assert.Empty(t, rules)
}

func assert35(t *testing.T, rules rule.Rules) {
	t.Helper()
}

func TestOneLineConfig(t *testing.T) {
	rules := assertLoadConfig(t, "*.*.*:NAMESPACE:KEY = 35;\n")
	assert35(t, rules)
}

func TestOneLineConfigCommentBefore(t *testing.T) {
	rules := assertLoadConfig(t, "//A Comment\n*.*.*:NAMESPACE:KEY = 35;\n")
	assert35(t, rules)
}

func TestOneLineConfigCommentAfter(t *testing.T) {
	rules := assertLoadConfig(t, "*.*.*:NAMESPACE:KEY = 35;//Set it to 35\n")
	assert35(t, rules)
}

func TestTwoLineConfig(t *testing.T) {
	rules := assertLoadConfig(t, "*.*.*:NAMESPACE:KEY = 35;\n*.*.*:Namespace:Key = 45;")

	assert.Equal(t, len(rules), 2)
	rule := rules[0]
	assert.Equal(t, rule.Selectors, []string{"*", "*", "*"})
	assert.Equal(t, rule.Namespace, "NAMESPACE")
	assert.Equal(t, rule.Keys, []string{"KEY"})
	assert.Equal(t, *rule.Value.IntValue, int64(35))
	assert.Equal(t, rule.Operator, OperatorEqual)

	rule = rules[1]
	assert.Equal(t, rule.Selectors, []string{"*", "*", "*"})
	assert.Equal(t, rule.Namespace, "Namespace")
	assert.Equal(t, rule.Keys, []string{"Key"})
	assert.Equal(t, *rule.Value.IntValue, int64(45))
	assert.Equal(t, rule.Operator, OperatorEqual)
}

func TestDifferentScoping(t *testing.T) {
	rules := assertLoadConfig(t, "Hats.us-west-2.production:nsamazon:keh = \"42\";\n")

	assert.Equal(t, len(rules), 1)
	rule := rules[0]
	assert.Equal(t, rule.Selectors, []string{"Hats", "us-west-2", "production"})
	assert.Equal(t, rule.Namespace, "nsamazon")
	assert.Equal(t, rule.Keys, []string{"keh"})
	assert.Equal(t, *rule.Value.StrValue, "42")
}

func TestIntValueFromConfig(t *testing.T) {
	rules := assertLoadConfig(t, "*.*.*:ns:key = 9999999999;")

	assert.Equal(t, *rules[0].Value.IntValue, int64(9999999999))
}

func TestStringValue(t *testing.T) {
	rules := assertLoadConfig(t, "*.*.*:ns:key = \"string\";")
	assert.Equal(t, *rules[0].Value.StrValue, "string")
}

func TestBoolValue(t *testing.T) {
	rules := assertLoadConfig(t, "*.*.*:ns:key = false;")
	assert.False(t, *rules[0].Value.BoolValue)
}

func TestDurationValueMillis(t *testing.T) {
	rules := assertLoadConfig(t, "*:ns:key = 300ms;")
	assert.NotNil(t, rules[0].Value.DurationValue)
	assert.Equal(t, int64(300*1000000), rules[0].Value.DurationValue.Nanoseconds())
	assert.Equal(t, float64(0.300), rules[0].Value.DurationValue.Seconds())
}

func TestDurationValueSeconds(t *testing.T) {
	rules := assertLoadConfig(t, "*:ns:key = 300s;")
	assert.NotNil(t, rules[0].Value.DurationValue)
	assert.Equal(t, float64(300), rules[0].Value.DurationValue.Seconds())
}

func TestDurationValueMinutes(t *testing.T) {
	rules := assertLoadConfig(t, "*:ns:key = 300m;")
	assert.NotNil(t, rules[0].Value.DurationValue)
	assert.Equal(t, float64(300), rules[0].Value.DurationValue.Minutes())
}

func TestDurationValueHours(t *testing.T) {
	rules := assertLoadConfig(t, "*:ns:key = 300h;")
	assert.NotNil(t, rules[0].Value.DurationValue)
	assert.Equal(t, float64(300), rules[0].Value.DurationValue.Hours())
}

func TestMultipleKeys(t *testing.T) {
	rules := assertLoadConfig(t, "*.*.*:\n\tNAMESPACE:KEY = 35;\n\tNamespace:Key = 45;")

	assert.Equal(t, len(rules), 2)
	rule := rules[0]
	assert.Equal(t, rule.Selectors, []string{"*", "*", "*"})
	assert.Equal(t, rule.Namespace, "NAMESPACE")
	assert.Equal(t, rule.Keys, []string{"KEY"})
	assert.Equal(t, *rule.Value.IntValue, int64(35))
	assert.Equal(t, rule.Operator, OperatorEqual)

	rule = rules[1]
	assert.Equal(t, rule.Selectors, []string{"*", "*", "*"})
	assert.Equal(t, rule.Namespace, "Namespace")
	assert.Equal(t, rule.Keys, []string{"Key"})
	assert.Equal(t, *rule.Value.IntValue, int64(45))
	assert.Equal(t, rule.Operator, OperatorEqual)
}

func TestRulesString(t *testing.T) {
	rules := assertLoadConfig(t, "*.*.*:NAMESPACE:KEY = 35;\n*.*.*:Namespace:Key = 45;")
	assert.Equal(t, `*.*.*:NAMESPACE:KEY = 35;
*.*.*:Namespace:Key = 45;
`, rules.String())
}

func TestRuleWithUnderscore(t *testing.T) {
	strKey := "*.*.*:NAMESPACE:key_with_underscore = 35;\n"
	rules := assertLoadConfig(t, strKey)
	assert.Equal(t, strKey, rules.String())
}
