package rule

import (
	"github.com/stretchr/testify/assert"
	"testing"
	"time"
)

func TestInt(t *testing.T) {
	i := int64(35)
	rule := &Rule{
		Selectors: []string{"*", "*", "*"},
		Keys:      []string{"k1", "k2"},
		Namespace: "ns",
		Value:     C7Value{IntValue: &i},
	}

	assert.Equal(t, rule.String(), "*.*.*:ns:k1.k2 = 35;")
	assert.Equal(t, rule.PrefixString(), "*.*.*:ns:k1.k2")
	assert.Equal(t, rule.StrType(), "int64")
}

func TestString(t *testing.T) {
	s := "hi"
	rule := &Rule{
		Selectors: []string{"a", "b", "c"},
		Keys:      []string{"k"},
		Namespace: "ns2",
		Value:     C7Value{StrValue: &s},
	}

	assert.Equal(t, rule.String(), "a.b.c:ns2:k = \"hi\";")

	s2 := "low"
	r2 := &Rule{
		Selectors: rule.Selectors,
		Keys:      rule.Keys,
		Namespace: rule.Namespace,
		Value:     C7Value{StrValue: &s2},
	}
	assert.True(t, rule.EqualsIgnoringValue(r2))

	r2 = &Rule{
		Selectors: rule.Selectors,
		Keys:      append(rule.Keys, "k2"),
		Namespace: rule.Namespace,
		Value:     C7Value{StrValue: rule.Value.StrValue},
	}
	assert.False(t, rule.EqualsIgnoringValue(r2))

	r2 = &Rule{
		Selectors: append(rule.Selectors, "s2"),
		Keys:      rule.Keys,
		Namespace: rule.Namespace,
		Value:     C7Value{StrValue: rule.Value.StrValue},
	}
	assert.False(t, rule.EqualsIgnoringValue(r2))

	r2 = &Rule{
		Selectors: rule.Selectors,
		Keys:      rule.Keys,
		Namespace: rule.Namespace + "a",
		Value:     C7Value{StrValue: rule.Value.StrValue},
	}
	assert.False(t, rule.EqualsIgnoringValue(r2))

	assert.Equal(t, "string", rule.StrType())
}

func TestBool(t *testing.T) {
	b := false
	rule := Rule{
		Selectors: []string{"a", "b", "c"},
		Keys:      []string{"k"},
		Namespace: "ns2",
		Value:     C7Value{BoolValue: &b},
	}
	assert.Equal(t, "a.b.c:ns2:k = false;", rule.String())
}

func TestDurationMilliseconds(t *testing.T) {
	d, _ := time.ParseDuration("300ms")
	rule := Rule{
		Selectors: []string{"a", "b", "c"},
		Keys:      []string{"k"},
		Namespace: "ns2",
		Value:     C7Value{DurationValue: &d},
	}
	assert.Equal(t, "a.b.c:ns2:k = 300ms;", rule.String())
}

func TestDurationSeconds(t *testing.T) {
	d, _ := time.ParseDuration("301s")
	rule := Rule{
		Selectors: []string{"a", "b", "c"},
		Keys:      []string{"k"},
		Namespace: "ns2",
		Value:     C7Value{DurationValue: &d},
	}
	assert.Equal(t, "a.b.c:ns2:k = 301s;", rule.String())
}

func TestDurationMinutes(t *testing.T) {
	d, _ := time.ParseDuration("301m")
	rule := Rule{
		Selectors: []string{"a", "b", "c"},
		Keys:      []string{"k"},
		Namespace: "ns2",
		Value:     C7Value{DurationValue: &d},
	}
	assert.Equal(t, "a.b.c:ns2:k = 301m;", rule.String())
}

func TestDurationHours(t *testing.T) {
	d, _ := time.ParseDuration("300h")
	rule := Rule{
		Selectors: []string{"a", "b", "c"},
		Keys:      []string{"k"},
		Namespace: "ns2",
		Value:     C7Value{DurationValue: &d},
	}
	assert.Equal(t, "a.b.c:ns2:k = 300h;", rule.String())
}

func TestMatches(t *testing.T) {
	rule := Rule{
		Selectors: []string{"a", "b", "c"},
		Keys:      []string{"k"},
		Namespace: "ns2",
	}
	assert.True(t, rule.Matches("ns2", "k"))
}

func TestMatchesRuleDifferentNamespace(t *testing.T) {
	rule := Rule{
		Selectors: []string{"a", "b", "c"},
		Keys:      []string{"k"},
		Namespace: "ns2",
	}
	assert.False(t, rule.Matches("ns3", "k"))
}

func TestMatchesWithDifferentKeys(t *testing.T) {
	rule := Rule{
		Selectors: []string{"a", "b", "c"},
		Keys:      []string{"ks"},
		Namespace: "ns2",
	}
	assert.False(t, rule.Matches("ns2", "k"))
}
