package specfile

import (
	"errors"
	"fmt"
	"io"
	"regexp"
	"strings"
)

// Spec contains one line from a specfile, which is generally called tuples.json, and is how
// we automatically test our configurations to see if they will load.
type Spec struct {
	Typestr    string   `json:"type"`
	Namespaces []string `json:"namespaces"`
	Selectors  []string `json:"selectors"`
	Index      int
}

// ImportPath returns the standard go path to this spec object
func (s *Spec) ImportPath() string {
	re := regexp.MustCompile(`(.+)\.([^\.]+)`)
	match := re.FindStringSubmatch(s.Typestr)
	return match[1]
}

// ImportLine returns the line as it should be in the import statements for this spec
func (s *Spec) ImportLine() string {
	return fmt.Sprintf("i%d \"%s\"", s.Index, s.ImportPath())
}

// Type the variable declaration type for this spec
func (s *Spec) Type() string {
	re := regexp.MustCompile(`.+\.([^\.]+)`)
	t := re.FindStringSubmatch(s.Typestr)[1]
	return fmt.Sprintf("i%d.%s", s.Index, t)
}

// TestName name for the test method
func (s *Spec) TestName() string {
	re := regexp.MustCompile(`.+\.([^\.]+)`)
	t := re.FindStringSubmatch(s.Typestr)[1]
	return fmt.Sprintf("i%d%s", s.Index, t)
}

// Validate checks if the spec seems sane
func (s *Spec) Validate() error {
	var errs []string
	if strings.Count(s.Typestr, ".") < 1 {
		errs = append(errs, "Spec type must be of the format importPath.type")
	}
	if s.Namespaces == nil || len(s.Namespaces) <= 0 {
		errs = append(errs, "Must specify a namespace")
	}
	if s.Selectors == nil || len(s.Selectors) <= 0 {
		errs = append(errs, "Must specify selectors")
	}
	if len(errs) > 0 {
		return errors.New(strings.Join(errs, " and "))
	}
	return nil
}

// OutputSpecTests ouputs a golang test to the location
func OutputSpecTests(packageName string, specs []*Spec, f io.Writer, configFile string) {
	for i, spec := range specs {
		spec.Index = i
	}
	fmt.Fprintf(f, `// +build !integration
	
package %s

import (
  "code.justin.tv/amzn/C7-go/c7"
  "code.justin.tv/amzn/C7-go/resolvers"
  "os"
  "testing"

`, packageName)

	// gather imports
	imports := make(map[string]*Spec)
	for _, spec := range specs {
		imports[spec.ImportPath()] = spec
	}

	// output imports
	for _, spec := range imports {
		fmt.Fprintf(f, "  %s\n", spec.ImportLine()) // spaces not tabs, we're not animals
	}
	fmt.Fprintf(f, `)

const (
  TestFilePath = "%s"
)

func testFile() string {
  if os.Getenv("C7_TEST_CONFIG") != "" {
    return os.Getenv("C7_TEST_CONFIG")
  }
  return TestFilePath
}

func resolveFile(t *testing.T) *c7.C7Set {
  set, err := resolvers.ResolveFile(testFile())
  if err != nil {
    t.Fatalf("Error resolving C7 %%s: %%s", testFile(), err)
  }

  if set == nil {
    t.Fatal("Set was nil")
  }

  return set
}
`, configFile)

	// Generate tests
	for _, spec := range specs {
		for _, selector := range spec.Selectors {
			for _, namespace := range spec.Namespaces {
				safeSelector := strings.Replace(selector, ".", "_", -1)
				safeSelector = strings.Replace(safeSelector, "-", "", -1)
				selectorParts := strings.Split(selector, ".")
				fmt.Fprintf(f, `
func Test_%s_%s_%s(t *testing.T) {
  set := resolveFile(t)
  cfg := c7.NewC7(*set, "%s")

  var config %s
  err := cfg.FillWithNamespace("%s", &config)
  if err != nil {
`,
					spec.TestName(),
					safeSelector,
					namespace,
					strings.Join(selectorParts, `", "`),
					spec.Type(),
					namespace)
				fmt.Fprintf(f, "    t.Fatalf(`Error loading config for %s", spec.Typestr)
				fmt.Fprintf(f, `
  Selectors: %s
  Namespace: %s
  Error: %%s`,
					selector,
					namespace)
				fmt.Fprint(f, "`, err)")
				fmt.Fprint(f, `
  }
}
`)
			}
		}
	}
}
