package specfile_test

import (
	"bytes"
	"code.justin.tv/amzn/C7-go/internal/specfile"
	"encoding/json"
	"fmt"
	"github.com/stretchr/testify/assert"
	"io/ioutil"
	"os"
	"os/exec"
	"testing"
)

func TestParseStandardPackage(t *testing.T) {
	strJson := `{
    "type": "golang.a2z.com/FultonGolangTemplateLambda/c7s.Config",
    "selectors": ["us-west-2.production"],
    "namespaces": ["FultonGolangTemplateLambda"]
  }`
	var spec specfile.Spec
	err := json.Unmarshal([]byte(strJson), &spec)
	assert.NoError(t, err)
	assert.NotNil(t, spec)

	spec.Index = 0

	assert.NoError(t, spec.Validate())
	assert.Equal(t, []string{"us-west-2.production"}, spec.Selectors)
	assert.Equal(t, []string{"FultonGolangTemplateLambda"}, spec.Namespaces)
	assert.Equal(t, "i0 \"golang.a2z.com/FultonGolangTemplateLambda/c7s\"", spec.ImportLine())
	assert.Equal(t, "golang.a2z.com/FultonGolangTemplateLambda/c7s", spec.ImportPath())
	assert.Equal(t, "i0.Config", spec.Type())
	assert.Equal(t, "i0Config", spec.TestName())
}

func TestOutputSpecTests(t *testing.T) {
	specs := []*specfile.Spec{
		&specfile.Spec{
			Typestr:    "golang.a2z.com/FultonGolangTemplateLambda/c7s.Config",
			Namespaces: []string{"FultonGolangTemplateLambda"},
			Selectors:  []string{"us-west-2.production"},
		},
		&specfile.Spec{
			Typestr:    "code.justin.tv/amzn/TwirpGoLangClientAutowire/autowire.TwirpClientConfig",
			Namespaces: []string{"StalkMarket"},
			Selectors:  []string{"us-west-2.development"},
		},
	}
	expected := `// +build !integration
	
package c7s_test

import (
  "code.justin.tv/amzn/C7-go/c7"
  "code.justin.tv/amzn/C7-go/resolvers"
  "os"
  "testing"

  i0 "golang.a2z.com/FultonGolangTemplateLambda/c7s"
  i1 "code.justin.tv/amzn/TwirpGoLangClientAutowire/autowire"
)

const (
  TestFilePath = "../c7/main.c7"
)

func testFile() string {
  if os.Getenv("C7_TEST_CONFIG") != "" {
    return os.Getenv("C7_TEST_CONFIG")
  }
  return TestFilePath
}

func resolveFile(t *testing.T) *c7.C7Set {
  set, err := resolvers.ResolveFile(testFile())
  if err != nil {
    t.Fatalf("Error resolving C7 %s: %s", testFile(), err)
  }

  if set == nil {
    t.Fatal("Set was nil")
  }

  return set
}

func Test_i0Config_uswest2_production_FultonGolangTemplateLambda(t *testing.T) {
  set := resolveFile(t)
  cfg := c7.NewC7(*set, "us-west-2", "production")

  var config i0.Config
  err := cfg.FillWithNamespace("FultonGolangTemplateLambda", &config)
  if err != nil {
    t.Fatalf(` + "`" + `Error loading config for golang.a2z.com/FultonGolangTemplateLambda/c7s.Config
  Selectors: us-west-2.production
  Namespace: FultonGolangTemplateLambda
  Error: %s` + "`" + `, err)
  }
}

func Test_i1TwirpClientConfig_uswest2_development_StalkMarket(t *testing.T) {
  set := resolveFile(t)
  cfg := c7.NewC7(*set, "us-west-2", "development")

  var config i1.TwirpClientConfig
  err := cfg.FillWithNamespace("StalkMarket", &config)
  if err != nil {
    t.Fatalf(` + "`" + `Error loading config for code.justin.tv/amzn/TwirpGoLangClientAutowire/autowire.TwirpClientConfig
  Selectors: us-west-2.development
  Namespace: StalkMarket
  Error: %s` + "`" + `, err)
  }
}
`
	w := bytes.Buffer{}
	specfile.OutputSpecTests("c7s_test", specs, &w, "../c7/main.c7")
	actual := string(w.Bytes())
	if expected != actual {
		if os.Getenv("DIFF") != "" {
			eF, err := ioutil.TempFile("", "")
			assert.NoError(t, err)
			fmt.Fprint(eF, expected)
			eF.Close()

			aF, err := ioutil.TempFile("", "")
			assert.NoError(t, err)
			fmt.Fprint(aF, actual)
			aF.Close()

			exec.Command("bcompare", eF.Name(), aF.Name()).Output()
		}

		t.Fatalf(
			"Expected:\n[%s]\n to equal \n[%s]",
			expected,
			actual)
	}
}
