package resolvers_test

import (
	"testing"

	"path"

	"code.justin.tv/amzn/C7-go/c7"
	"code.justin.tv/amzn/C7-go/resolvers"
	"github.com/stretchr/testify/assert"
)

func TestSingleDirectory(t *testing.T) {
	set, err := resolvers.ResolveFilesWithExtension(path.Join("..", "examples", "files", "config"), ".acg")
	assert.NoError(t, err)
	assert.NotNil(t, set)

	cfg := c7.NewC7(*set, "HATS", "us-west-2", "development")

	type HatsConfig struct {
		String string `c7:"SandstormRoleARN"`
		Bool   bool   `c7:"StatsNoop"`
	}

	type SiteDBConfig struct {
		Int64 int64 `c7:"Port"`
	}

	hatsConfig := &HatsConfig{}
	err = cfg.FillWithNamespace("HATS", hatsConfig)
	assert.NoError(t, err)

	assert.Equal(t, "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/hats-dev", hatsConfig.String)
	assert.True(t, hatsConfig.Bool)

	sdbConfig := &SiteDBConfig{}
	err = cfg.FillWithNamespace("SiteDB", sdbConfig)
	assert.NoError(t, err)

	assert.Equal(t, int64(6543), sdbConfig.Int64)
}

func TestSimpleTreeNoConflicts(t *testing.T) {
	set, err := resolvers.ResolveFilesWithExtension(path.Join("..", "examples", "tree"), ".acg")
	assert.NoError(t, err)
	assert.NotNil(t, set)

	cfg := c7.NewC7(*set, "dev")
	type NSConfig struct {
		String string `c7:"string"`
		Bool   bool   `c7:"bool"`
		Int64  int64  `c7:"int64"`
	}

	nsConfig := &NSConfig{}
	err = cfg.FillWithNamespace("ns", nsConfig)
	assert.NoError(t, err)

	assert.False(t, nsConfig.Bool)
	assert.Equal(t, "hi", nsConfig.String)
	assert.Equal(t, int64(1), nsConfig.Int64)

	nsConfig2 := &NSConfig{}
	err = cfg.FillWithNamespace("ns2", nsConfig2)
	assert.NoError(t, err)

	assert.True(t, nsConfig2.Bool)
	assert.Equal(t, "yo", nsConfig2.String)
	assert.Equal(t, int64(2), nsConfig2.Int64)
}

func TestSingleFile(t *testing.T) {
	set, err := resolvers.ResolveFile(path.Join("..", "examples", "simple", "single.acg"))
	assert.NoError(t, err)
	assert.NotNil(t, set)

	cfg := c7.NewC7(*set, "dev", "a", "b")
	type Config struct {
		Int64_1 int64 `c7:"key"`
		Int64_2 int64 `c7:"key2"`
	}

	config := &Config{}
	err = cfg.FillWithNamespace("ns", config)
	assert.NoError(t, err)

	assert.Equal(t, int64(2), config.Int64_1)
	assert.Equal(t, int64(3), config.Int64_2)
}

func TestBrokenFile(t *testing.T) {
	acs, err := resolvers.ResolveFile(path.Join("..", "examples", "broken.c7"))
	assert.Error(t, err)
	assert.Nil(t, acs)
}
