package client

import (
	"bytes"
	"context"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"github.com/golang/protobuf/proto"

	serviceapi "code.justin.tv/amzn/PDMSLambdaTwirp"
	"code.justin.tv/amzn/PDMSLambdaUploadSchema/information"
)

// NewUploadClientFromTwirp create a new Upload helper that wraps an existing PDMSLambdaTwirp client
func NewUploadClientFromTwirp(client serviceapi.PDMSService) *UploadClient {
	return &UploadClient{
		PDMSService: client,
		httpClient: &http.Client{
			Timeout: time.Second * 60,
		},
	}
}

type UploadClient struct {
	serviceapi.PDMSService
	httpClient *http.Client
}

// Parameters to report a file
type ReportSingleFileRequest struct {
	// The DataID processing the access request
	DataID information.DataID
	// the access request id from the eventbus message
	RequestID string
	// the payload to upload
	Payload proto.Message
}

// UploadSingleFile upload a single file to PDMS, if the Payload is nil, then report NoData to PDMS
// if there is something to upload, call PDMS to then generate a s3 presigned url and then upload the data
func (client *UploadClient) ReportSingleFile(ctx context.Context, request ReportSingleFileRequest) error {
	if request.Payload == nil {
		_, err := client.PDMSService.ReportAccess(ctx, &serviceapi.ReportAccessRequest{
			RequestId: request.RequestID,
			DataId:    string(request.DataID),
			HasData:   false,
		})
		return err
	}

	resp, err := client.PDMSService.ReportAccess(ctx, &serviceapi.ReportAccessRequest{
		RequestId: request.RequestID,
		DataId:    string(request.DataID),
		HasData:   true,
	})
	if err != nil {
		return err
	}
	return client.uploadFile(resp.UploadUrl, request.Payload)
}

// uploadFile Marshal a proto message and then preform a http PUT to the s3 preassigned url
func (client *UploadClient) uploadFile(target string, message proto.Message) error {
	payload, err := proto.Marshal(message)
	if err != nil {
		return err
	}
	req, err := http.NewRequest("PUT", target, bytes.NewReader(payload))
	if err != nil {
		return err
	}

	resp, err := client.httpClient.Do(req)
	if err != nil {
		return err
	}

	if resp.StatusCode != http.StatusOK {
		body, err := ioutil.ReadAll(resp.Body)
		if err != nil {
			return err
		}
		return fmt.Errorf("bad http status: [%s] %s", resp.Status, string(body))
	}
	return nil
}

//ReportMultipleFilesRequest parameters needed to upload multi files
type ReportMultipleFilesRequest struct {
	// The DataID processing the access request
	DataID information.DataID
	// the access request id from the eventbus message
	RequestID string
	// the set of payloads map filename -> message to upload
	Payloads map[string]proto.Message
}

// UploadMultipleFiles use the multifile upload process to upload many files to PDMS access
// if Payloads is empty or nil, Report no data and exit.
func (client *UploadClient) ReportMultipleFiles(ctx context.Context, request ReportMultipleFilesRequest) error {
	if request.Payloads == nil || len(request.Payloads) == 0 {
		_, err := client.PDMSService.ReportAccess(ctx, &serviceapi.ReportAccessRequest{
			RequestId: request.RequestID,
			DataId:    string(request.DataID),
			HasData:   false,
		})
		return err
	}

	fileNames := make([]string, len(request.Payloads))
	payloads := make([]proto.Message, len(request.Payloads))
	i := 0
	for key, value := range request.Payloads {
		fileNames[i] = key
		payloads[i] = value
		i++
	}

	resp, err := client.PDMSService.GenerateMultiFileUrls(ctx, &serviceapi.GenerateMultiFileUploadLinksRequest{
		DataId:    string(request.DataID),
		RequestId: request.RequestID,
		FileNames: fileNames,
	})
	if err != nil {
		return err
	}

	for i, url := range resp.UploadUrls {
		err := client.uploadFile(url, payloads[i])
		if err != nil {
			return err
		}
	}

	_, err = client.CompleteMultiFileAccess(ctx, &serviceapi.CompleteMultiFileAccessRequest{
		ExpectedNumberFiles: int64(len(request.Payloads)),
		DataId:              string(request.DataID),
		RequestId:           request.RequestID,
	})
	return err
}
