package information

import (
	"encoding/json"
	"fmt"

	"github.com/golang/protobuf/proto"

	"code.justin.tv/amzn/PDMSLambdaUploadSchema/generated/datasets/chat/whispersv1"
	"code.justin.tv/amzn/PDMSLambdaUploadSchema/generated/datasets/connections/connectionsv1"
	"code.justin.tv/amzn/PDMSLambdaUploadSchema/generated/datasets/example/examplev1"
	"code.justin.tv/amzn/PDMSLambdaUploadSchema/generated/datasets/owl/authorizationsv1"
	"code.justin.tv/amzn/PDMSLambdaUploadSchema/generated/datasets/passport/passportv1"
	"code.justin.tv/amzn/PDMSLambdaUploadSchema/generated/datasets/users/usersv1"
)

type DataID string

const (
	Example  DataID = "example-data-id"
	UserInfo DataID = "user-information"

	ChatWhispers             DataID = "chat-whispers"
	ChatWhispersUserSettings DataID = "chat-whispers-settings"

	ConnectionsBlizzard DataID = "connections-blizzard"
	ConnectionsRiot     DataID = "connections-riot"
	ConnectionsSteam    DataID = "connections-steam"
	ConnectionsTwitter  DataID = "connections-twitter"
	ConnectionsYoutube  DataID = "connections-youtube"

	OwlAuthorization DataID = "owl-authorization"
	PassportAuthyID  DataID = "passport-authy-id"
)

// MetaInfo contains a mapping of all known DataID types to their MetaInformation definitions
type MetaInfo struct {
	Info map[DataID]MetaInformation
}

// MetaInformation of a piece of data
type MetaInformation struct {
	// ServiceID is the id of the service that owns the data, from the service catalog
	ServiceID string
	// Title is the title for the type of data being reported by your service
	Title string
	// Description describes the fields being reported in the objects uploaded by your service
	Description string
	// MessageFactory should return an object of the type of the proto message that we will deserialize the uploaded data into
	MessageFactory func() proto.Message
	// ReportLocation describes a path for the uploaded files and determines where the files will be located in the
	// report we provide to the user when it is prepared for them.
	ReportLocation string
	// PreProd will tell PDMS to only create access requests in dev for testing. Remove this flag once
	// a service has moved into production and is ready to receive events
	PreProd bool
	// DataID
	DataID DataID
}

// satisfy the Data interface{} for PDMS
func (mi MetaInformation) ReportPath() string {
	return mi.ReportLocation
}

func (mi MetaInformation) ReportDescription() string {
	return mi.Description
}

func (mi MetaInformation) PreProduction() bool {
	return mi.PreProd
}

func (mi MetaInformation) Category() string {
	return string(mi.DataID)
}

// Decode takes a DataID and a slice of bytes and returns the MetaInformation associated with the DataID and the pretty-
// printed JSON of the protobuf object associated with the DataID after populating it with the contents of the byte
// slice.
func (i *MetaInfo) Decode(dataID DataID, body []byte) (MetaInformation, []byte, error) {
	dataInfo, ok := i.Info[dataID]
	if !ok {
		return MetaInformation{}, nil, fmt.Errorf("unknown dataID")
	}

	// create a new message and populate it from the contents of body
	message := dataInfo.MessageFactory()
	err := proto.Unmarshal(body, message)
	if err != nil {
		return MetaInformation{}, nil, fmt.Errorf("unmarshalling body into proto message: %w", err)
	}

	// marshal it into a pretty-print json
	val, err := json.MarshalIndent(message, "", "  ")
	if err != nil {
		return MetaInformation{}, nil, fmt.Errorf("marshalling message into json: %w", err)
	}
	return dataInfo, val, nil
}

// GetInfo returns the list of report types that have been registered with PDMS.
// data id's need to be unique
func GetInfo() *MetaInfo {
	return &MetaInfo{
		Info: map[DataID]MetaInformation{
			// example-data-id is used for tests
			Example: {
				ServiceID:      "-1",
				Title:          "Example Title",
				Description:    "example description of the data",
				MessageFactory: func() proto.Message { return &examplev1.Person{} },
				ReportLocation: "/notused/",
				DataID:         Example,
			},
			UserInfo: {
				ServiceID:      "3",
				Title:          "User Information",
				Description:    "The account ID, email, birthday, and phone number",
				MessageFactory: func() proto.Message { return &usersv1.User{} },
				ReportLocation: "/account/",
				PreProd:        true,
				DataID:         UserInfo,
			},
			ChatWhispers: {
				// https://catalog.xarth.tv/services/45/details
				ServiceID:   "45",
				Title:       "User's one-on-one conversations with others",
				Description: "A set of conversations between the user and other users",
				// A UserThread is one entire conversation between two users
				MessageFactory: func() proto.Message { return &whispersv1.UserThread{} },
				ReportLocation: "/chat-whispers/",
				DataID:         ChatWhispers,
			},
			ChatWhispersUserSettings: {
				// https://catalog.xarth.tv/services/45/details
				ServiceID:      "45",
				Title:          "A user's whispers settings",
				Description:    "A user's whispers settings",
				MessageFactory: func() proto.Message { return &whispersv1.UserSettings{} },
				ReportLocation: "/chat-whispers-settings/",
				DataID:         ChatWhispersUserSettings,
			},
			OwlAuthorization: {
				// https://catalog.xarth.tv/services/11/details
				ServiceID:      "11",
				Title:          "Authorization from Owl",
				Description:    "Authorized 3rd party applications of Twitch OAuth 2.0 granted by a Twitch User",
				MessageFactory: func() proto.Message { return &authorizationsv1.Authorization{} },
				ReportLocation: "/owl-authorization/",
				PreProd:        true,
				DataID:         OwlAuthorization,
			},
			// passport authy id
			PassportAuthyID: {
				// https://catalog.xarth.tv/services/86/details
				ServiceID:   "86",
				Title:       "Twitch authentication portal",
				Description: "A set of conversations between the user and other users",
				// A UserThread is one entire conversation between two users
				MessageFactory: func() proto.Message { return &passportv1.TwoFactor{} },
				ReportLocation: "/authentication/two-factor/",
				PreProd:        true,
				DataID:         PassportAuthyID,
			},
			ConnectionsBlizzard: {
				// https://catalog.xarth.tv/services/4/details
				ServiceID:      "4",
				Title:          "Blizzard Connection",
				Description:    "A user's Blizzard account connection",
				MessageFactory: func() proto.Message { return &connectionsv1.BlizzardUser{} },
				ReportLocation: "/account/external-connections/",
				PreProd:        true,
				DataID:         ConnectionsBlizzard,
			},
			ConnectionsRiot: {
				// https://catalog.xarth.tv/services/4/details
				ServiceID:      "4",
				Title:          "Riot Connection",
				Description:    "A user's Riot account connection",
				MessageFactory: func() proto.Message { return &connectionsv1.RiotUser{} },
				ReportLocation: "/account/external-connections/",
				PreProd:        true,
				DataID:         ConnectionsRiot,
			},
			ConnectionsSteam: {
				// https://catalog.xarth.tv/services/4/details
				ServiceID:      "4",
				Title:          "Steam Connection",
				Description:    "A user's Steam account connection",
				MessageFactory: func() proto.Message { return &connectionsv1.SteamUser{} },
				ReportLocation: "/account/external-connections/",
				PreProd:        true,
				DataID:         ConnectionsSteam,
			},
			ConnectionsTwitter: {
				// https://catalog.xarth.tv/services/4/details
				ServiceID:      "4",
				Title:          "Twitter Connection",
				Description:    "A user's Twitter account connection",
				MessageFactory: func() proto.Message { return &connectionsv1.TwitterUser{} },
				ReportLocation: "/account/external-connections/",
				PreProd:        true,
				DataID:         ConnectionsTwitter,
			},
			ConnectionsYoutube: {
				// https://catalog.xarth.tv/services/4/details
				ServiceID:      "4",
				Title:          "Youtube Connection",
				Description:    "A user's Youtube account connection",
				MessageFactory: func() proto.Message { return &connectionsv1.YoutubeUser{} },
				ReportLocation: "/account/external-connections/",
				PreProd:        true,
				DataID:         ConnectionsYoutube,
			},
		},
	}
}
