package ReservoirTwirp

import (
	"fmt"
	"log"
	"net"
	"regexp"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/video/sflow"
)

// NewTimestamp creates an *sgaugeapi.Timestamp from a time.Time
func NewTimestamp(t time.Time) *Timestamp {
	return &Timestamp{
		Seconds: t.Unix(),
		Nanos:   int32(t.Nanosecond()),
	}
}

// NowTimestamp returns the current time as an *sgaugeapi.Timestamp
func NowTimestamp() *Timestamp {
	return NewTimestamp(time.Now())
}

func (ts *Timestamp) ToGoTime() time.Time {
	return time.Unix(ts.Seconds, int64(ts.Nanos))
}

// Returns a TraceMsg with Msg set to x.
//
// Protobuf wraps the actual type with an intermediate type that implements the
// interface we want. So this function will take the useable struct and convert
// it into the interface.
func NewMsg(x interface{}) *TraceMsg {
	var m isTraceMsg_Msg

	switch x := x.(type) {
	default:
		log.Fatalf("sgaugeapi.NewMsg: unexpected type %T\n", x)
	case *TraceStart:
		m = &TraceMsg_TraceStart{x}
	case *TraceFinishUnclean:
		m = &TraceMsg_TraceFinishUnclean{x}
	case *TraceEvResultIn:
		m = &TraceMsg_TraceEvResultIn{x}
	case *TraceEvResultsUsed:
		m = &TraceMsg_TraceEvResultsUsed{x}
	case *TraceFinishClean:
		m = &TraceMsg_TraceFinishClean{x}
	}

	//m := x.(isTraceMsg_Msg)
	return &TraceMsg{Msg: m}
}

// DiscoverAF will return what address family a net.IP belongs to.
func DiscoverAF(ip net.IP) IPVersion {
	v4q := ip.To4()

	if v4q != nil {
		return IPVersion_IPv4
	}

	return IPVersion_IPv6
}

// ToIPAddress will convert a net.IP to a to an sgaugeapi.IPAddress.
func ToIPAddress(netIP net.IP) *IPAddress {
	ip := &IPAddress{}
	ip.Type = DiscoverAF(netIP)
	ip.Address = []byte(netIP)
	return ip
}

// ToNetIP will convert a sgaugeapi.IPAddress to a to net.IP.
func (sip *IPAddress) ToNetIP() net.IP {
	return net.IP(sip.Address)
}

// ToCommunity will convert a sflow.Community to a sgaugeapi.Community.
func ToCommunity(sc sflow.Community) Community {
	return Community{
		ASN: uint32(sc.AS),
		Tag: uint32(sc.Tag),
	}
}

// NewCommunity returns a new sgaugeapi.Community. Its sole purpose is to make
// `go vet` shut the hell up about composite literals.
func NewCommunity(asn uint32, tag uint32) Community {
	return Community{
		ASN: asn,
		Tag: tag,
	}
}

// PrettyString is a helper function for printing a community as "ASN:Tag"
// string. We can't overwrite the c.String() function due to its
// internal use as part of protobuf protocol.
func (c *Community) PrettyString() string {
	return fmt.Sprintf("%d:%d", c.ASN, c.Tag)
}

// GenBucketName will generate the bucket name based upon what data is in the
// sample. This function can freely change in between runs since none of this
// data is serialized.
func GenBucketName(sample *Sample, includeSourceID bool) string {
	// This code is optimized to do as few allocations as possible. It does
	// this by preallocating a scratch byte array to work in. It then does
	// appends and binary writes directly to the buffer before finally
	// returning the string representation of the buffer.
	if sample == nil || sample.Origin == nil || sample.Origin.AgentIP == nil {
		return ""
	}

	arr := make([]byte, 0, 128)

	for i := range sample.Origin.AgentIP.Address {
		arr = strconv.AppendUint(arr, uint64(sample.Origin.AgentIP.Address[i]), 10)
		if i < len(sample.Origin.AgentIP.Address)-1 {
			arr = append(arr, '.')
		}
	}

	arr = append(arr, '-')
	arr = strconv.AppendUint(arr, uint64(sample.Origin.SubAgentID), 10)

	if includeSourceID {
		arr = append(arr, '-')
		arr = strconv.AppendUint(arr, uint64(sample.Origin.SourceID), 10)
	}

	return string(arr)
}

// DuplicateSample will create an entirely new sample in memory with identical
// data.
func DuplicateSample(sample *Sample) *Sample {
	newSample := &Sample{}

	// Copy all fields
	*newSample = *sample

	if sample.Community != nil {
		newSample.Community = &Community{
			ASN: sample.Community.ASN,
			Tag: sample.Community.Tag,
		}
	}

	if sample.Communities != nil {
		newSample.Communities = make([]*Community, 0, len(sample.Communities))
		for _, community := range sample.Communities {
			newSample.Communities = append(newSample.Communities, &Community{
				ASN: community.ASN,
				Tag: community.Tag,
			})
		}
	}

	if sample.Origin != nil && sample.Origin.AgentIP != nil {
		newIP := &IPAddress{
			Address: make([]byte, len(sample.Origin.AgentIP.Address)),
			Type:    sample.Origin.AgentIP.Type,
		}

		copy(newIP.Address, sample.Origin.AgentIP.Address)

		newSample.Origin = &Origin{
			AgentIP:    newIP,
			SourceID:   sample.Origin.SourceID,
			SubAgentID: sample.Origin.SubAgentID,
		}
	}

	if sample.CollectorTimestamp != nil {
		newSample.CollectorTimestamp = &Timestamp{
			Seconds: sample.CollectorTimestamp.Seconds,
			Nanos:   sample.CollectorTimestamp.Nanos,
		}
	}

	return newSample
}

var validateCommunityRE = regexp.MustCompile("^[0-9]+:[0-9]+$")

// ParseCommunity will attempt to parse a community, e.g. 46489:12001. If it fails it will return
// an error. Because the returned Community is not a pointer you **must** check
// if error is nil or not.
func ParseCommunity(input string) (Community, error) {
	if !validateCommunityRE.MatchString(input) {
		return Community{}, fmt.Errorf("Attempted to parse %q, does not appear to be a valid `<asn>:<tag>` community", input)
	}

	splitInput := strings.SplitN(input, ":", 3)
	if len(splitInput) != 2 {
		return Community{}, fmt.Errorf("Splitting %q on ':' did not return the expected two part", input)
	}

	asn, err := strconv.ParseInt(splitInput[0], 10, 32)
	if err != nil {
		return Community{}, fmt.Errorf("Error parsing asn from %q: %v", input, err)
	}

	tag, err := strconv.ParseInt(splitInput[1], 10, 32)
	if err != nil {
		return Community{}, fmt.Errorf("Error parsing tag from %q: %v", input, err)
	}

	return Community{ASN: uint32(asn), Tag: uint32(tag)}, nil
}
