package ReservoirTwirp

import (
	"net"
	"reflect"
	"testing"
)

var s1 = &Sample{
	Community: &Community{
		ASN: 65534,
		Tag: 50,
	},
	CollectorTimestamp: &Timestamp{},

	OriginTimestamp: 63,
	PacketID:        15,
	SampleID:        15,
	SampleRate:      8192,
	Length:          1200,
	Drops:           5,

	Origin: &Origin{
		AgentIP: &IPAddress{
			Address: []byte{192, 168, 1, 1},
			Type:    0,
		},
		SourceID:   15,
		SubAgentID: 5,
	},
}

func TestDuplicateSample(t *testing.T) {
	s2 := DuplicateSample(s1)

	if s1 == nil || s2 == nil {
		t.Fatalf("s1 or s2 are equal to nil")
	}

	if s2 == s1 {
		t.Errorf("Address of s2 == s1")
	}

	if s2.Community == s1.Community {
		t.Errorf("Address of s2.Community == s1.Community")
	}

	if !reflect.DeepEqual(s1, s2) {
		t.Errorf("reflect.DeepEqual(s1, s2) != true")
	}
}

func BenchmarkDuplicateSample(b *testing.B) {
	for i := 0; i < b.N; i++ {
		_ = DuplicateSample(s1)
	}
}

func TestParseCommunity(t *testing.T) {
	type testCase struct {
		input     string
		community Community
		shouldErr bool
	}

	empty := Community{}

	testCases := []testCase{
		{"46489:12001", Community{ASN: 46489, Tag: 12001}, false},
		{"46489_12001", empty, true},
	}

	for _, c := range testCases {
		community, err := ParseCommunity(c.input)

		if (err == nil) == c.shouldErr {
			t.Errorf("ParseCommunity(%q) should error: %v; got error: %v", c.input, c.shouldErr, err)
			continue
		}

		if c.community.ASN != community.ASN || c.community.Tag != community.Tag {
			t.Errorf("ParseCommunity(%q) = %v; expected %v", c.input, community, c.community)
		}
	}
}

func TestDiscoverAF(t *testing.T) {
	type testCase struct {
		ip net.IP
		af IPVersion
	}

	testCases := []testCase{
		{net.ParseIP("192.168.1.2"), IPVersion_IPv4},
		{net.ParseIP("::ffff:1234:5678"), IPVersion_IPv4}, // RFC4291 s. 2.5.5.2
		{net.ParseIP("fe80::1"), IPVersion_IPv6},
	}

	for _, c := range testCases {
		discoveredAF := DiscoverAF(c.ip)
		if discoveredAF != c.af {
			t.Errorf("DiscoverAF(%v) = %v; expected %v", c.ip, discoveredAF, c.af)
		}
	}
}
