package cloudwatch

import (
	"context"
	"encoding/json"
	"os"

	"code.justin.tv/amzn/StarfruitAbyss/internal/models"
	"github.com/aws/aws-lambda-go/events"
	"github.com/aws/aws-lambda-go/lambda"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	log "github.com/sirupsen/logrus"
)

var (
	// get event name from environment. this is set in cloudformation
	eventType       = os.Getenv("EVENT_TYPE")
	cwClient        = cloudwatch.New(session.New(aws.NewConfig()))
	metricNamespace = "AbyssProcessor"
)

func init() {
	log.SetFormatter(&log.JSONFormatter{})
	log.SetOutput(os.Stdout)
}

const (
	batchSize = 20
)

func Run() {
	lambda.Start(run)
}

type metricDatumer interface {
	MetricDatum() ([]*cloudwatch.MetricDatum, error)
}

func run(ctx context.Context, event events.KinesisEvent) error {
	// list for all records
	var metrics []*cloudwatch.MetricDatum

	for _, record := range event.Records {
		var md metricDatumer

		// unmarshal event by event type
		switch eventType {
		case "Contribute":
			var c models.ContentContributed
			if err := json.Unmarshal(record.Kinesis.Data, &c); err != nil {
				log.WithError(err).Error("failed to unmarshal content contributed")
				continue
			}
			md = c
		case "Transcode":
			var t models.ContentTranscoded
			if err := json.Unmarshal(record.Kinesis.Data, &t); err != nil {
				log.WithError(err).Error("failed to unmarshal content transcoded")
				continue
			}
			md = t
		case "Deliver":
			var d models.ContentDelivered
			if err := json.Unmarshal(record.Kinesis.Data, &d); err != nil {
				log.WithError(err).Error("failed to unmarshal content delivered")
				continue
			}
			md = d
		}

		datums, err := md.MetricDatum()
		if err != nil {
			log.WithError(err).Error("failed to make metric datum")
			continue
		}

		// add event to list
		metrics = append(metrics, datums...)
	}

	// send cloudwatch metric datums in batches
	for i := 0; i < len(metrics); i += batchSize {
		end := i + batchSize

		if end > len(metrics) {
			end = len(metrics)
		}

		// send to cloudwatch
		_, err := cwClient.PutMetricDataWithContext(ctx, &cloudwatch.PutMetricDataInput{
			MetricData: metrics[i:end],
			Namespace:  aws.String(metricNamespace),
		})
		if err != nil {
			log.WithError(err).Error("failed to post cloudwatch metrics")
		}
	}

	return nil
}
