package dummy

import (
	"encoding/json"
	"log"
	"os"
	"strconv"
	"time"

	"github.com/aws/aws-lambda-go/lambda"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/firehose"
	"github.com/pkg/errors"
)

var periodInMinutes, _ = strconv.Atoi(os.Getenv("PeriodInMinutes"))
var firehoseContributedStream = os.Getenv("FirehoseContributeStreamName")
var firehoseTranscodedStream = os.Getenv("FirehoseTranscodeStreamName")
var firehoseDeliveredStream = os.Getenv("FirehoseDeliverStreamName")

var rxBytesRange = intRange{1000, 3000}
var bytesDeliveredRange = intRange{1200000, 1700000}
var msgJitterMillisRange = intRange{0, 100}
var distribStartJitterSecRange = intRange{10, 60}
var distribLengthMinRange = intRange{5, periodInMinutes - 1}

var sess = session.Must(session.NewSession())
var firehoseService = firehose.New(sess, aws.NewConfig())

func Run() {
	if os.Getenv("ENVIRONMENT") == "local" {
		spawnStreams()
		return
	}

	lambda.Start(spawnStreams)
}

func logError(err error) {
	if err == nil {
		return
	}

	log.Println(err)
}

func emit(v interface{}, streamName string) error {
	j, err := json.Marshal(v)
	if err != nil {
		return errors.Wrap(err, "failed to marshal message JSON")
	}
	record := &firehose.Record{Data: j}
	recordInput := &firehose.PutRecordInput{
		DeliveryStreamName: aws.String(streamName),
		Record:             record,
	}
	_, err = firehoseService.PutRecord(recordInput)
	return errors.Wrap(err, "failed to put Firehose record")
}

func spawnStreams() (string, error) {
	const numBroadcasts = 1
	done := make(chan bool)

	var broadcasts []*broadcast
	for i := 0; i < numBroadcasts; i++ {
		broadcasts = append(broadcasts, newIngest())
	}

	const numDistributionStreams = 3
	for _, broadcast := range broadcasts {
		go startNewIngestStream(broadcast, done)

		for i := 0; i < numDistributionStreams; i++ {
			go startNewDistributionStream(broadcast)
		}
	}
	time.Sleep(time.Duration(periodInMinutes)*time.Minute - (30 * time.Second))
	close(done)
	return "Finished", nil
}

func startNewIngestStream(b *broadcast, done chan bool) {
	period := 2 * time.Second
	ticker := time.NewTicker(period)
	go func() {
		for {
			select {
			case <-done:
				ticker.Stop()
				return
			case <-ticker.C:
				go func() {
					jitterMillis(msgJitterMillisRange)
					logError(emit(b.contribute(), firehoseContributedStream))
				}()
				go func() {
					jitterMillis(msgJitterMillisRange)
					logError(emit(b.transcode(), firehoseTranscodedStream))
				}()
			}
		}
	}()
}

func startNewDistributionStream(b *broadcast) {
	period := 2 * time.Second
	jitterSec(distribStartJitterSecRange)
	ticker := time.NewTicker(period)
	done := make(chan bool)
	go func() {
		for {
			select {
			case <-done:
				ticker.Stop()
				return
			case <-ticker.C:
				go func() {
					jitterMillis(msgJitterMillisRange)
					logError(emit(b.delivered(period), firehoseDeliveredStream))
				}()
			}
		}
	}()
	streamLength := distribLengthMinRange.Sample()
	time.Sleep(time.Duration(streamLength) * time.Minute)
	close(done)
}

func jitterMillis(r intRange) {
	time.Sleep(time.Duration(r.Sample()) * time.Millisecond)
}

func jitterSec(r intRange) {
	time.Sleep(time.Duration(r.Sample()) * time.Second)
}
