package producer

import (
	"context"
	"fmt"

	abyss "code.justin.tv/amzn/AwsStarfruitAbyssCollectorTwirp"
	"github.com/aws/aws-sdk-go/service/sqs/sqsiface"
	"github.com/pkg/errors"
)

// AbyssProducer is a client to ship data to the Abyss.
//
// TODO: Add batch APIs.
type AbyssProducer struct {
	stage       string
	boundRegion string
	sqs         sqsiface.SQSAPI
}

// NewAbyssProducers creates an AbyssProducer client which sends data to the
// Abyss. It sends it for a particular stage ('prod', 'beta', or 'gamma'), and
// to queues housed within a particular bound region ('us-west-2', 'us-east-2').
func NewAbyssProducer(stage string, boundRegion string, sqs sqsiface.SQSAPI) (*AbyssProducer, error) {
	err := validateProducerParams(stage, boundRegion)
	if err != nil {
		return nil, errors.Wrap(err, "AbyssProducer parameters are invalid")
	}

	return &AbyssProducer{
		stage:       stage,
		boundRegion: boundRegion,
		sqs:         sqs,
	}, nil
}

// RecordContentTranscoded sends a single ContentTranscoded message into the Abyss.
func (p *AbyssProducer) RecordContentTranscoded(ctx context.Context, msg *abyss.ContentTranscoded) error {
	homeRegion, err := HomeRegionForChannel(msg.GetStream().GetStream().ChannelArn)
	if err != nil {
		return errors.Wrap(err, "unable to determine home region for channel")
	}

	qurl, err := queueURL(p.stage, p.boundRegion, homeRegion, "Transcode")
	if err != nil {
		return errors.Wrap(err, "unable to determine destination queue for message")
	}

	batchMsg := &abyss.RecordContentTranscodedRequest{
		Data: []*abyss.ContentTranscoded{msg},
	}

	protoProducer := NewProtoSQSProducer(qurl, p.sqs)
	return protoProducer.Send(ctx, batchMsg)
}

// RecordContentContributed sends a single ContentContributed message into the Abyss.
func (p *AbyssProducer) RecordContentContributed(ctx context.Context, msg *abyss.ContentContributed) error {
	homeRegion, err := HomeRegionForChannel(msg.GetStream().GetStream().ChannelArn)
	if err != nil {
		return errors.Wrap(err, "unable to determine home region for channel")
	}

	qurl, err := queueURL(p.stage, p.boundRegion, homeRegion, "Contribute")
	if err != nil {
		return errors.Wrap(err, "unable to determine destination queue for message")
	}

	batchMsg := &abyss.RecordContentContributedRequest{
		Data: []*abyss.ContentContributed{msg},
	}

	protoProducer := NewProtoSQSProducer(qurl, p.sqs)
	return protoProducer.Send(ctx, batchMsg)
}

// RecordContentDelivered sends a single ContentDelivered message into the Abyss.
func (p *AbyssProducer) RecordContentDelivered(ctx context.Context, msg *abyss.ContentDelivered) error {
	homeRegion, err := HomeRegionForChannel(msg.GetStream().ChannelArn)
	if err != nil {
		return errors.Wrap(err, "unable to determine home region for channel")
	}

	qurl, err := queueURL(p.stage, p.boundRegion, homeRegion, "Deliver")
	if err != nil {
		return errors.Wrap(err, "unable to determine destination queue for message")
	}

	batchMsg := &abyss.RecordContentDeliveredRequest{
		Data: []*abyss.ContentDelivered{msg},
	}

	protoProducer := NewProtoSQSProducer(qurl, p.sqs)
	return protoProducer.Send(ctx, batchMsg)
}

// RecordContentDeliveredBatch sends a collection of *abyss.ContentDelivered
// messages to the Abyss. All of the messages should be destined for the same
// home region. The appropriate home region for a message can be determined
// using the HomeRegionForChannel function. The messages must not exceed
// MaxPayloadSize bytes when encoded; their encoded length can be determined
// using the MessageSize function.
func (p *AbyssProducer) RecordContentDeliveredBatch(ctx context.Context, homeRegion string, msg *abyss.RecordContentDeliveredRequest) error {
	err := validateHomeRegions(homeRegion, msg.Data)
	if err != nil {
		return err
	}

	qurl, err := queueURL(p.stage, p.boundRegion, homeRegion, "Deliver")
	if err != nil {
		return errors.Wrap(err, "unable to determine destination queue for message")
	}

	protoProducer := NewProtoSQSProducer(qurl, p.sqs)
	return protoProducer.Send(ctx, msg)
}

// validateHomeRegions makes sure that all the given messages have valid channel
// ARNs, and that all ARNs have the same home region as the given input
// parameter. It returns a nil error if everything looks good.
func validateHomeRegions(homeRegion string, messages []*abyss.ContentDelivered) error {
	for _, msg := range messages {
		arn := msg.GetStream().ChannelArn
		have, err := HomeRegionForChannel(arn)
		if err != nil {
			return errors.Wrapf(err, "unable to determine home region for channel ARN %q", arn)
		}
		if have != homeRegion {
			return fmt.Errorf("batch contains a message with arn %q, which has home region %q, not %q", arn, have, homeRegion)
		}
	}
	return nil
}
