package producer

import (
	"context"
	"encoding/base64"
	"fmt"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/aws/aws-sdk-go/service/sqs/sqsiface"
	"github.com/golang/protobuf/proto"
	"github.com/pkg/errors"
)

// MaxPayloadSize is the maximum permitted size in bytes of a message (measured
// with the MessageSize function) to be sendable in SQS.
const MaxPayloadSize = (250 * 1000)

// MessageSize returns the size of a message if it were to be encoded into a
// payload for transmission in SQS. Valid messages should not exceed
// MaxPayloadSize.
func MessageSize(msg proto.Message) int {
	// proto.Size is cheap in modern editions of generated code from
	// protoc-gen-go, since it produces a XXX_Size() int method on all generated
	// objects which gets called, so this doens't require fully marshaling the
	// message.
	return base64.StdEncoding.EncodedLen(proto.Size(msg))
}

// ErrorMessageTooBig is an error indicating that a message cannot be sent to
// SQS because it is too large when encoded.
type ErrorMessageTooBig struct {
	// Size is the encoded size of the message.
	Size int
	// Message is the message that was too large to encode.
	Message proto.Message
}

func (e ErrorMessageTooBig) Error() string {
	return fmt.Sprintf("message of length %d bytes exceeds maximum sendable size", e.Size)
}

// A ProtoSQSProducer encodes proto messages for transport in SQS, and sends
// them on a specific queue.
//
// This is a low-level client. Most users will want to use AbyssProducer.
type ProtoSQSProducer struct {
	sqs      sqsiface.SQSAPI
	queueURL string
}

func NewProtoSQSProducer(queueURL string, sqs sqsiface.SQSAPI) *ProtoSQSProducer {
	return &ProtoSQSProducer{queueURL: queueURL, sqs: sqs}
}

func (p *ProtoSQSProducer) Encode(msg proto.Message) (*sqs.SendMessageInput, error) {
	size := MessageSize(msg)
	if size > MaxPayloadSize {
		return nil, ErrorMessageTooBig{Size: size, Message: msg}
	}
	b, err := proto.Marshal(msg)
	if err != nil {
		return nil, err
	}

	body := base64.StdEncoding.EncodeToString(b)
	v := &sqs.SendMessageInput{
		QueueUrl:    aws.String(p.queueURL),
		MessageBody: aws.String(body),
	}

	return v, nil
}

func (p *ProtoSQSProducer) Send(ctx context.Context, payload proto.Message) error {
	req, err := p.Encode(payload)
	if err != nil {
		return errors.Wrap(err, "unable to encode message")
	}
	_, err = p.sqs.SendMessageWithContext(ctx, req)
	if err != nil {
		return errors.Wrap(err, "unable to send message to SQS")
	}
	return nil
}
