package producer

import (
	"testing"
	"time"

	abyss "code.justin.tv/amzn/AwsStarfruitAbyssCollectorTwirp"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/golang/protobuf/ptypes"
	"github.com/google/uuid"
)

func TestProtoSQSProducerEncode(t *testing.T) {
	const queueURL = "dummy-queue"

	id := uuid.New()
	msg := &abyss.ContentTranscoded{
		Meta: &abyss.AbyssEventMeta{
			Uuid: id[:],
			Time: ptypes.TimestampNow(),
		},
		Stream: &abyss.ContributionStream{
			BroadcastSessionId: "session-1",
			Stream: &abyss.StreamResource{
				ChannelArn: "arn:aws:starfruit:us-west-2:123456780000:channel:testdata",
			},
		},
		Length:         ptypes.DurationProto(2 * time.Second),
		TranscoderType: abyss.TranscoderType_X264,
		TranscodeRenditions: []*abyss.TranscodeRendition{
			&abyss.TranscodeRendition{Label: "chunked"},
			&abyss.TranscodeRendition{Label: "720p60"},
			&abyss.TranscodeRendition{Label: "720p30"},
			&abyss.TranscodeRendition{Label: "480p30"},
			&abyss.TranscodeRendition{Label: "audio_only"},
		},
		OriginDc: "sjc02",
	}
	producer := NewProtoSQSProducer(queueURL, nil)

	v, err := producer.Encode(msg)
	if err != nil {
		t.Fatalf("encode error: %v", err)
	}

	if have := aws.StringValue(v.QueueUrl); have != queueURL {
		t.Errorf("invalid encoded QueueURL  have=%q  want=%q", have, queueURL)
	}

	if v.MessageBody == nil {
		t.Fatal("unexpected nil MessageBody after encoding")
	}

	msgBody := *v.MessageBody
	if len(msgBody) == 0 {
		t.Error("unexpected empty string for MessageBody")
	}
	if len(msgBody) > MaxPayloadSize {
		t.Errorf("MessageBody of size %d exceeds max of %d", len(msgBody), MaxPayloadSize)
	}
}

func TestProtoSQSProducerEncodeLargeMessage(t *testing.T) {
	const queueURL = "dummy-queue"

	bigString := string(make([]byte, 2*MaxPayloadSize))
	msg := &abyss.ContentTranscoded{Stream: &abyss.ContributionStream{BroadcastSessionId: bigString}}
	producer := NewProtoSQSProducer(queueURL, nil)

	_, err := producer.Encode(msg)
	if err == nil {
		t.Fatal("unexpected nil error (expected ErrorMessageTooBig")
	}

	if _, ok := err.(ErrorMessageTooBig); !ok {
		t.Errorf("unexpected error type  have=%T  want=ErrorMessageTooBig", err)
	}
}
