package producer

import (
	"fmt"
	"strings"

	"github.com/aws/aws-sdk-go/aws/arn"
)

// Map of stage -> bound region -> ID of account hosting SQS queues
var accountIDs = map[string]map[string]string{
	"prod": map[string]string{
		"us-west-2": "500600498289",
		"us-east-2": "422782966314",
	},
	"beta": map[string]string{
		"us-west-2": "717518721197",
		"us-east-2": "118171646437",
	},
}

func validateProducerParams(stage, boundRegion string) error {
	stageMap, ok := accountIDs[stage]
	if !ok {
		return fmt.Errorf("stage %q has no queues", stage)
	}

	_, ok = stageMap[boundRegion]
	if !ok {
		return fmt.Errorf("queues are not set up for %q stage in %q region", stage, boundRegion)
	}

	return nil
}

func queueURL(stage, boundRegion, homeRegion, message string) (string, error) {
	err := validateProducerParams(stage, boundRegion)
	if err != nil {
		return "", err
	}
	accountID := accountIDs[stage][boundRegion]
	queueName := fmt.Sprintf("bound-region-%s-%s", strings.ToLower(message), homeRegion)
	url := fmt.Sprintf("https://sqs.%s.amazonaws.com/%s/%s", boundRegion, accountID, queueName)
	return url, nil
}

// HomeRegionForChannel returns the home region ID where the channel's data
// should be sent (for example "us-west-2"). If the channel ARN is not parsable,
// this may return an error.
func HomeRegionForChannel(channelARN string) (string, error) {
	parsed, err := arn.Parse(channelARN)
	if err != nil {
		return "", err
	}
	return parsed.Region, nil
}
