package producer

import "testing"

func TestHomeRegionForChannel(t *testing.T) {
	testcase := func(channelARN, want string, wantErr bool) func(*testing.T) {
		return func(t *testing.T) {
			have, err := HomeRegionForChannel(channelARN)
			if wantErr {
				if err == nil {
					t.Fatalf("expected an error, but its nil (got region=%q)", have)
				}
				return
			}

			if err != nil {
				t.Fatalf("unexpected err: %v", err)
			}

			if have != want {
				t.Error("wrong region")
				t.Logf("have=%q", have)
				t.Logf("want=%q", want)
			}
		}
	}

	t.Run("empty", testcase("", "", true))
	t.Run("not an arn", testcase("lirik", "", true))
	t.Run("no region", testcase("arn:aws:starfruit::123456789012:channel/whatever", "", false))
	t.Run("valid arn", testcase("arn:aws:starfruit:us-west-2:123456789012:channel/whatever", "us-west-2", false))
}

func TestQueueURL(t *testing.T) {
	testcase := func(stage, boundRegion, homeRegion, message string, wantURL string, wantErr bool) func(*testing.T) {
		return func(t *testing.T) {
			haveURL, err := queueURL(stage, boundRegion, homeRegion, message)
			if wantErr {
				if err == nil {
					t.Fatalf("expected an error, but its nil (got url=%q)", haveURL)
				}
				return
			}

			if err != nil {
				t.Fatalf("unexpected err: %v", err)
			}

			if haveURL != wantURL {
				t.Error("wrong URL")
				t.Logf("have=%q", haveURL)
				t.Logf("want=%q", wantURL)
			}
		}
	}

	t.Run("valid prod",
		testcase(
			"prod", "us-west-2", "us-east-1", "Contribute",
			"https://sqs.us-west-2.amazonaws.com/500600498289/bound-region-contribute-us-east-1",
			false),
	)

	t.Run("valid beta",
		testcase(
			"beta", "us-east-2", "us-east-1", "Contribute",
			"https://sqs.us-east-2.amazonaws.com/118171646437/bound-region-contribute-us-east-1",
			false),
	)

	t.Run("bad stage",
		testcase(
			"staging", "us-east-2", "us-east-1", "Contribute",
			"", true),
	)

	t.Run("bad bound region",
		testcase(
			"prod", "potato", "us-east-1", "Contribute",
			"", true),
	)
}
