package StarfruitAbyssProducer

import (
	"context"
	"fmt"
	"time"

	abyss "code.justin.tv/amzn/AwsStarfruitAbyssCollectorTwirp"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/arn"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/golang/protobuf/ptypes"
	"github.com/pkg/errors"
)

var (
	ErrInvalidStreamID = errors.New("stream ID cannot be zero")
)

// AbyssProducer is a client to ship data to the Abyss.
//
// TODO: Add batch APIs.
type AbyssProducer struct {
	stage          string
	producerRegion string
	sqs            SQSAPI
}

type SQSAPI interface {
	SendMessageWithContext(aws.Context, *sqs.SendMessageInput, ...request.Option) (*sqs.SendMessageOutput, error)
}

// NewAbyssProducers creates an AbyssProducer client which sends data to the
// Abyss. It sends it for a particular stage ('prod', 'beta', or 'gamma'), and
// to queues housed within a particular producer region ('us-west-2', 'us-east-2', 'eu-west-2', 'eu-west-1', 'us-east-1').
func NewAbyssProducer(stage string, producerRegion string, sqs SQSAPI) (*AbyssProducer, error) {
	_, err := queueAccount(stage, producerRegion)
	if err != nil {
		return nil, errors.Wrap(err, "AbyssProducer parameters are invalid")
	}

	return &AbyssProducer{
		stage:          stage,
		producerRegion: producerRegion,
		sqs:            sqs,
	}, nil
}

// RecordContentTranscoded sends a single ContentTranscoded message into the Abyss.
// If any of the fields in msg are empty or invalid, the message will not be recorded
// and RecordContentTranscoded will return an error.
func (p *AbyssProducer) RecordContentTranscoded(ctx context.Context, msg *abyss.ContentTranscoded) error {
	err := validateContentTranscoded(msg)
	if err != nil {
		return err
	}

	homeRegion, err := HomeRegionForChannel(msg.GetStream().GetStream().ChannelArn)
	if err != nil {
		return errors.Wrap(err, "unable to determine home region for channel")
	}

	qurl, err := queueURL(p.stage, p.producerRegion, homeRegion, transcode)
	if err != nil {
		return errors.Wrap(err, "unable to determine destination queue for message")
	}

	batchMsg := &abyss.RecordContentTranscodedRequest{
		Data: []*abyss.ContentTranscoded{msg},
	}

	protoProducer := NewProtoSQSProducer(qurl, p.sqs)
	return protoProducer.Send(ctx, batchMsg)
}

// RecordContentContributed sends a single ContentContributed message into the Abyss.
// If any of the fields in msg are empty or invalid, the message will not be recorded
// and RecordContentContributed will return an error.
func (p *AbyssProducer) RecordContentContributed(ctx context.Context, msg *abyss.ContentContributed) error {
	err := validateContentContributed(msg)
	if err != nil {
		return err
	}

	homeRegion, err := HomeRegionForChannel(msg.GetStream().GetStream().ChannelArn)
	if err != nil {
		return errors.Wrap(err, "unable to determine home region for channel")
	}

	qurl, err := queueURL(p.stage, p.producerRegion, homeRegion, contribute)
	if err != nil {
		return errors.Wrap(err, "unable to determine destination queue for message")
	}

	batchMsg := &abyss.RecordContentContributedRequest{
		Data: []*abyss.ContentContributed{msg},
	}

	protoProducer := NewProtoSQSProducer(qurl, p.sqs)
	return protoProducer.Send(ctx, batchMsg)
}

// RecordContentDelivered sends a single ContentDelivered message into the Abyss.
// If any of the fields in msg are empty or invalid, the message will not be recorded
// and RecordContentDelivered will return an error.
func (p *AbyssProducer) RecordContentDelivered(ctx context.Context, msg *abyss.ContentDelivered) error {
	err := validateContentDelivered([]*abyss.ContentDelivered{msg})
	if err != nil {
		return err
	}

	homeRegion, err := HomeRegionForChannel(msg.GetStream().ChannelArn)
	if err != nil {
		return errors.Wrap(err, "unable to determine home region for channel")
	}

	qurl, err := queueURL(p.stage, p.producerRegion, homeRegion, deliver)
	if err != nil {
		return errors.Wrap(err, "unable to determine destination queue for message")
	}

	batchMsg := &abyss.RecordContentDeliveredRequest{
		Data: []*abyss.ContentDelivered{msg},
	}

	protoProducer := NewProtoSQSProducer(qurl, p.sqs)
	return protoProducer.Send(ctx, batchMsg)
}

// RecordContentDeliveredBatch sends a collection of *abyss.ContentDelivered
// messages to the Abyss. All of the messages should be destined for the same
// home region. The appropriate home region for a message can be determined
// using the HomeRegionForChannel function. The messages must not exceed
// MaxPayloadSize bytes when encoded; their encoded length can be determined
// using the MessageSize function.
//
// If any messages in the batch request have an empty or invalid field,
// RecordContentDeliveredBatch will fail the entire batch and return an error.
func (p *AbyssProducer) RecordContentDeliveredBatch(ctx context.Context, homeRegion string, msg *abyss.RecordContentDeliveredRequest) error {
	err := validateContentDelivered(msg.Data)
	if err != nil {
		return err
	}

	err = validateHomeRegionsForContentDelivered(homeRegion, msg.Data)
	if err != nil {
		return err
	}
	qurl, err := queueURL(p.stage, p.producerRegion, homeRegion, deliver)
	if err != nil {
		return errors.Wrap(err, "unable to determine destination queue for message")
	}

	protoProducer := NewProtoSQSProducer(qurl, p.sqs)
	return protoProducer.Send(ctx, msg)
}

// RecordContentRecordedBatch sends a collection of *abyss.ContentRecorded
// messages to the Abyss. All of the messages should be destined for the same
// home region. The appropriate home region for a message can be determined
// using the HomeRegionForChannel function.
//
// If any messages in the batch request have an empty or invalid field,
// RecordContentRecordedBatch will fail the entire batch and return an error.
func (p *AbyssProducer) RecordContentRecordedBatch(ctx context.Context, homeRegion string, msg *abyss.RecordContentRecordedRequest) error {
	if msg.GetData() == nil {
		return nil
	}

	err := validateContentRecorded(msg.Data)
	if err != nil {
		return err
	}
	err = validateHomeRegionsForContentRecorded(homeRegion, msg.Data)
	if err != nil {
		return err
	}
	qurl, err := queueURL(p.stage, p.producerRegion, homeRegion, record)
	if err != nil {
		return errors.Wrap(err, "unable to determine destination queue for message")
	}
	protoProducer := NewProtoSQSProducer(qurl, p.sqs)
	return protoProducer.Send(ctx, msg)
}

// RecordContentRecorded sends a *abyss.ContentRecorded message to the Abyss.
// If any of the fields in msg are empty or invalid, the message will not be recorded
// and RecordContentRecorded will return an error.
func (p *AbyssProducer) RecordContentRecorded(ctx context.Context, msg *abyss.ContentRecorded) error {
	if msg.GetStream() == nil {
		return nil
	}

	err := validateContentRecorded([]*abyss.ContentRecorded{msg})
	if err != nil {
		return err
	}

	homeRegion, err := HomeRegionForChannel(msg.GetStream().GetChannelArn())
	if err != nil {
		return errors.Errorf("unable to determine home region for channel:%s, %v", msg.GetStream().GetChannelArn(), err)
	}

	qurl, err := queueURL(p.stage, p.producerRegion, homeRegion, record)
	if err != nil {
		return errors.Wrap(err, "unable to determine destination queue for message")
	}
	batchMsg := &abyss.RecordContentRecordedRequest{
		Data: []*abyss.ContentRecorded{msg},
	}
	protoProducer := NewProtoSQSProducer(qurl, p.sqs)
	return protoProducer.Send(ctx, batchMsg)
}

func (p *AbyssProducer) RecordConcurrentViewsBatch(ctx context.Context, homeRegion string, msg *abyss.RecordConcurrentViewsRequest) error {
	err := validateContentCCV(homeRegion, msg.Data)
	if err != nil {
		return err
	}

	qurl, err := queueURL(p.stage, p.producerRegion, homeRegion, ccv)
	if err != nil {
		return errors.Wrap(err, "unable to determine destination queue for message")
	}

	protoProducer := NewProtoSQSProducer(qurl, p.sqs)
	return protoProducer.Send(ctx, msg)
}

func (p *AbyssProducer) RecordConcurrentStreamsBatch(ctx context.Context, homeRegion string, msg *abyss.RecordConcurrentStreamsRequest) error {
	err := validateContentCCS(msg.Data)
	if err != nil {
		return err
	}

	qurl, err := queueURL(p.stage, p.producerRegion, homeRegion, ccs)
	if err != nil {
		return errors.Wrap(err, "unable to determine destination queue for message")
	}

	protoProducer := NewProtoSQSProducer(qurl, p.sqs)
	return protoProducer.Send(ctx, msg)
}

func (p *AbyssProducer) RecordMessageReceivedBatch(ctx context.Context, homeRegion string, msg *abyss.RecordMessageReceivedRequest) error {
	if err := validateMessageReceived(msg.Data); err != nil {
		return err
	}

	qurl, err := queueURL(p.stage, p.producerRegion, homeRegion, messageReceived)
	if err != nil {
		return errors.Wrap(err, "unable to determine destination queue for message")
	}

	protoProducer := NewProtoSQSProducer(qurl, p.sqs)
	return protoProducer.Send(ctx, msg)
}

func (p *AbyssProducer) RecordMessageDeliveredBatch(ctx context.Context, homeRegion string, msg *abyss.RecordMessageDeliveredRequest) error {
	if err := validateMessageDelivered(msg.Data); err != nil {
		return err
	}

	qurl, err := queueURL(p.stage, p.producerRegion, homeRegion, messageDelivered)
	if err != nil {
		return errors.Wrap(err, "unable to determine destination queue for message")
	}

	protoProducer := NewProtoSQSProducer(qurl, p.sqs)
	return protoProducer.Send(ctx, msg)
}

func (p *AbyssProducer) RecordConcurrentChatConnectionsBatch(ctx context.Context, homeRegion string, msg *abyss.RecordConcurrentChatConnectionsRequest) error {
	if err := validateChatCCC(msg.Data); err != nil {
		return err
	}

	qurl, err := queueURL(p.stage, p.producerRegion, homeRegion, ccc)
	if err != nil {
		return errors.Wrap(err, "unable to determine destination queue for message")
	}

	protoProducer := NewProtoSQSProducer(qurl, p.sqs)
	return protoProducer.Send(ctx, msg)
}

// validateHomeRegions makes sure that all the given messages have valid channel
// ARNs, and that all ARNs have the same home region as the given input
// parameter. It returns a nil error if everything looks good.
func validateHomeRegion(homeRegion string, arn string) error {
	have, err := HomeRegionForChannel(arn)
	if err != nil {
		return errors.Wrapf(err, "unable to determine home region for channel ARN %q", arn)
	}
	if have != homeRegion {
		return fmt.Errorf("batch contains a message with arn %q, which has home region %q, not %q", arn, have, homeRegion)
	}
	return nil
}

func validateHomeRegionsForContentDelivered(homeRegion string, messages []*abyss.ContentDelivered) error {
	for _, msg := range messages {
		err := validateHomeRegion(homeRegion, msg.GetStream().ChannelArn)
		if err != nil {
			return err
		}
	}
	return nil
}
func validateHomeRegionsForContentRecorded(homeRegion string, messages []*abyss.ContentRecorded) error {
	for _, msg := range messages {
		err := validateHomeRegion(homeRegion, msg.GetStream().ChannelArn)
		if err != nil {
			return err
		}
	}
	return nil
}

func validateContentTranscoded(msg *abyss.ContentTranscoded) error {
	ts, err := ptypes.Timestamp(msg.Meta.Time)
	if err != nil {
		return errors.Wrapf(err, "invalid timestamp %q", msg.Meta.Time)
	}
	if invalidTime(&ts) {
		return errors.Errorf("invalid timestamp %q", msg.Meta.Time)
	}
	_, err = ptypes.Duration(msg.Length)
	if err != nil {
		return errors.Wrapf(err, "invalid duration %q", msg.Length)
	}
	_, err = arn.Parse(msg.Stream.Stream.ChannelArn)
	if err != nil {
		return errors.Wrapf(err, "invalid channel ARN %q", msg.Stream.Stream.ChannelArn)
	}

	if msg.Stream.Stream.StreamId == 0 {
		return ErrInvalidStreamID
	}

	for _, segment := range msg.MediaSegments {
		if _, err := ptypes.Duration(segment.Duration); err != nil {
			return errors.Wrapf(err, "segment %d has invalid duration: %q", segment.SequenceNumber, err)
		}
	}

	return nil
}

func validateContentContributed(msg *abyss.ContentContributed) error {
	ts, err := ptypes.Timestamp(msg.Meta.Time)
	if err != nil {
		return errors.Wrapf(err, "invalid timestamp %q", msg.Meta.Time)
	}
	if invalidTime(&ts) {
		return errors.Errorf("invalid timestamp %q", msg.Meta.Time)
	}
	_, err = ptypes.Duration(msg.Length)
	if err != nil {
		return errors.Wrapf(err, "invalid duration %q", msg.Length)
	}
	_, err = arn.Parse(msg.Stream.Stream.ChannelArn)
	if err != nil {
		return errors.Wrapf(err, "invalid channel ARN %q", msg.Stream.Stream.ChannelArn)
	}
	return nil
}

func validateContentDelivered(messages []*abyss.ContentDelivered) error {
	for _, msg := range messages {
		ts, err := ptypes.Timestamp(msg.Meta.Time)
		if err != nil {
			return errors.Wrapf(err, "invalid timestamp %q", msg.Meta.Time)
		}
		if invalidTime(&ts) {
			return errors.Errorf("invalid timestamp %q", msg.Meta.Time)
		}
		_, err = ptypes.Duration(msg.ContentLength)
		if err != nil {
			return errors.Wrapf(err, "invalid duration %q", msg.ContentLength)
		}
		_, err = arn.Parse(msg.Stream.ChannelArn)
		if err != nil {
			return errors.Wrapf(err, "invalid channel ARN %q", msg.Stream.ChannelArn)
		}

		if msg.Stream.StreamId == 0 {
			return ErrInvalidStreamID
		}
	}
	return nil
}

func validateContentRecorded(messages []*abyss.ContentRecorded) error {
	for _, msg := range messages {
		ts, err := ptypes.Timestamp(msg.Meta.Time)
		if err != nil {
			return errors.Wrapf(err, "invalid timestamp %q", msg.Meta.Time)
		}
		if invalidTime(&ts) {
			return errors.Errorf("invalid timestamp %q", msg.Meta.Time)
		}
		_, err = ptypes.Duration(msg.Duration)
		if err != nil {
			return errors.Wrapf(err, "invalid duration %q", msg.Duration)
		}
		_, err = arn.Parse(msg.Stream.ChannelArn)
		if err != nil {
			return errors.Wrapf(err, "invalid channel ARN %q", msg.Stream.ChannelArn)
		}
	}
	return nil
}

func validateContentCCV(homeRegion string, data []*abyss.ConcurrentViews) error {
	for _, datum := range data {
		ts, err := ptypes.Timestamp(datum.Meta.Time)
		if err != nil {
			return errors.Wrapf(err, "invalid timestamp %q", datum.Meta.Time)
		}
		if invalidTime(&ts) {
			return errors.Errorf("invalid timestamp %q", datum.Meta.Time)
		}
		if datum.Stream == nil && datum.AccountId == "" {
			return fmt.Errorf("Unable to find identifying information for ccv datum")
		}
		if datum.Stream != nil {
			if datum.Stream.ChannelArn == "" {
				return fmt.Errorf("Unable to find a channel arn for ccv datum")
			}
			a, err := arn.Parse(datum.Stream.ChannelArn)
			if err != nil {
				return errors.Wrapf(err, "invalid channel ARN %q", datum.Stream.ChannelArn)
			}
			if a.Region != homeRegion {
				return fmt.Errorf("Found mismatch between regionality of client and data. client: %s, data: %s", homeRegion, a.Region)
			}
		}
	}
	return nil
}

func validateContentCCS(data []*abyss.ConcurrentStreams) error {
	for _, datum := range data {
		ts, err := ptypes.Timestamp(datum.Meta.Time)
		if err != nil {
			return errors.Wrapf(err, "invalid timestamp %q", datum.Meta.Time)
		}
		if invalidTime(&ts) {
			return errors.Errorf("invalid timestamp %q", datum.Meta.Time)
		}
		if datum.AccountId == "" {
			return fmt.Errorf("Unable to find an account id for ccb datum")
		}
	}
	return nil
}

func validateMessageReceived(data []*abyss.MessageReceived) error {
	for _, datum := range data {
		ts, err := ptypes.Timestamp(datum.Meta.Time)
		if err != nil {
			return errors.Wrapf(err, "invalid timestamp %q", datum.Meta.Time)
		}
		if invalidTime(&ts) {
			return errors.Errorf("invalid timestamp %q", datum.Meta.Time)
		}
		if datum.Room == nil {
			return errors.Errorf("invalid room %+v", datum.Room)
		}
		if _, err := arn.Parse(datum.Room.Arn); err != nil {
			return errors.Wrapf(err, "invalid room ARN %q", datum.Room.Arn)
		}

		switch datum.Type {
		case abyss.MessageType_ROOM_MESSAGE:
			err = validateRoomMessageReceived(datum)
		case abyss.MessageType_SYSTEM_MESSAGE:
		case abyss.MessageType_EVENT_MESSAGE:
			break
		case abyss.MessageType_DISCONNECT_USER:
		case abyss.MessageType_DELETE_MESSAGE:
			err = validateOperationReceived(datum)
		default:
			return errors.Errorf("invalid message type %q", abyss.MessageType_name[int32(datum.Type)])
		}

		if err != nil {
			return err
		}
	}
	return nil
}

func validateMessageDelivered(data []*abyss.MessageDelivered) error {
	for _, datum := range data {
		ts, err := ptypes.Timestamp(datum.Meta.Time)
		if err != nil {
			return errors.Wrapf(err, "invalid timestamp %q", datum.Meta.Time)
		}
		if invalidTime(&ts) {
			return errors.Errorf("invalid timestamp %q", datum.Meta.Time)
		}
		if datum.Room == nil {
			return errors.Errorf("invalid room %+v", datum.Room)
		}
		if _, err := arn.Parse(datum.Room.Arn); err != nil {
			return errors.Wrapf(err, "invalid room ARN %q", datum.Room.Arn)
		}
		if datum.MessageId == "" {
			return errors.Errorf("invalid MessageId")
		}
		if datum.Receiver == nil || datum.Receiver.Id == "" || datum.Receiver.Ip == "" {
			return errors.Errorf("invalid receiver %+v", datum.Receiver)
		}

		switch datum.Type {
		case abyss.MessageType_ROOM_MESSAGE:
			err = validateRoomMessageDelivered(datum)
		case abyss.MessageType_SYSTEM_MESSAGE:
		case abyss.MessageType_EVENT_MESSAGE:
		case abyss.MessageType_DISCONNECT_USER:
		case abyss.MessageType_DELETE_MESSAGE:
			break
		default:
			return errors.Errorf("invalid message type %q", abyss.MessageType_name[int32(datum.Type)])
		}

		if err != nil {
			return err
		}
	}
	return nil
}

func validateRoomMessageReceived(datum *abyss.MessageReceived) error {
	if datum.Sender == nil {
		return errors.New("sender is nil")
	}
	if datum.Sender.Id == "" {
		return errors.Errorf("invalid sender ID %q", datum.Sender.Id)
	}
	if datum.Sender.Ip == "" {
		return errors.Errorf("invalid sender IP %q", datum.Sender.Ip)
	}
	return nil
}

func validateOperationReceived(datum *abyss.MessageReceived) error {
	if datum.Sender == nil {
		return errors.New("sender is nil")
	}
	if datum.Sender.Id == "" {
		return errors.Errorf("invalid sender ID %q", datum.Sender.Id)
	}
	if datum.Sender.Ip == "" {
		return errors.Errorf("invalid sender IP %q", datum.Sender.Ip)
	}
	if datum.TargetId == "" {
		return errors.Errorf("invalid target ID %q", datum.TargetId)
	}

	return nil
}

func validateRoomMessageDelivered(datum *abyss.MessageDelivered) error {
	if datum.Sender == nil {
		return errors.New("sender is nil")
	}
	if datum.Sender.Id == "" {
		return errors.Errorf("invalid sender ID %q", datum.Sender.Id)
	}
	if datum.Sender.Ip == "" {
		return errors.Errorf("invalid sender IP %q", datum.Sender.Ip)
	}
	return nil
}

func validateChatCCC(data []*abyss.ConcurrentChatConnections) error {
	for _, datum := range data {
		ts, err := ptypes.Timestamp(datum.Meta.Time)
		if err != nil {
			return errors.Wrapf(err, "invalid timestamp %q", datum.Meta.Time)
		}
		if invalidTime(&ts) {
			return errors.Errorf("invalid timestamp %q", datum.Meta.Time)
		}
		if datum.Room == nil {
			return errors.Errorf("invalid room %+v", datum.Room)
		}
		if _, err := arn.Parse(datum.Room.Arn); err != nil {
			return errors.Wrapf(err, "invalid room ARN %q", datum.Room.Arn)
		}
	}
	return nil
}

// invalidTime returns true if the given time is equal to the zero value
//
// the ptypes.Timestamp function will convert to time.Unix(0, 0).UTC(),
// which comes out to 1970-01-01 00:00:00 +0000, which is not the golang
// zero-value for time (time.Time{} = 0001-01-01 00:00:00 +0000)
func invalidTime(t *time.Time) bool {
	unixZero := time.Unix(0, 0).UTC()
	return t.IsZero() || t.Equal(unixZero) || t.Before(unixZero)
}
