package StarfruitAbyssProducer

import (
	"context"
	"fmt"
	"testing"
	"time"

	abyss "code.justin.tv/amzn/AwsStarfruitAbyssCollectorTwirp"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/arn"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/golang/protobuf/ptypes"
	"github.com/golang/protobuf/ptypes/timestamp"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

var (
	testHomeRegion = "us-west-2"
	channelARN     = (arn.ARN{
		Region: "us-west-2",
	}).String()

	roomARN = (arn.ARN{
		Service:   "ivs",
		Region:    "us-west-2",
		AccountID: "testcustomer",
		Resource:  "testroom",
	}).String()

	now = &timestamp.Timestamp{
		Seconds: time.Now().Unix(),
	}

	zeroTime, _ = ptypes.TimestampProto(time.Time{})

	thirtySeconds = ptypes.DurationProto(30 * time.Second)
)

func TestAbyssProducer(t *testing.T) {
	fakeSQS := &fakeSQS{}

	producer, err := NewAbyssProducer("beta", "us-west-2", fakeSQS)
	require.NoError(t, err)

	ctx := context.Background()

	require.NoError(t, producer.RecordContentContributed(ctx, &abyss.ContentContributed{
		Meta: &abyss.AbyssEventMeta{
			Time: now,
		},
		Length: thirtySeconds,
		Stream: &abyss.ContributionStream{
			Stream: &abyss.StreamResource{
				ChannelArn: channelARN,
			},
		},
	}))
	require.Equal(t, 1, fakeSQS.SendMessageCallCount)

	require.NoError(t, producer.RecordContentTranscoded(ctx, &abyss.ContentTranscoded{
		Meta: &abyss.AbyssEventMeta{
			Time: now,
		},
		Length: thirtySeconds,
		Stream: &abyss.ContributionStream{
			Stream: &abyss.StreamResource{
				ChannelArn: channelARN,
				StreamId:   1,
			},
		},
		TranscodeRenditions: []*abyss.TranscodeRendition{
			{
				Label: "chunked",
			},
		},
		MediaSegments: []*abyss.MediaSegment{
			{
				SequenceNumber: 1,
				Duration:       ptypes.DurationProto(2 * time.Second),
			},
		},
	}))
	require.Equal(t, 2, fakeSQS.SendMessageCallCount)

	require.NoError(t, producer.RecordContentDelivered(ctx, &abyss.ContentDelivered{
		Meta: &abyss.AbyssEventMeta{
			Time: now,
		},
		ContentLength: thirtySeconds,
		Stream: &abyss.StreamResource{
			ChannelArn: channelARN,
			StreamId:   1,
		},
	}))
	require.Equal(t, 3, fakeSQS.SendMessageCallCount)

	require.NoError(t, producer.RecordContentRecorded(ctx, &abyss.ContentRecorded{
		Meta: &abyss.AbyssEventMeta{
			Time: now,
		},
		Duration: thirtySeconds,
		Stream: &abyss.StreamResource{
			ChannelArn: channelARN,
		},
	}))
	require.Equal(t, 4, fakeSQS.SendMessageCallCount)

	require.NoError(t, producer.RecordContentDeliveredBatch(ctx, "us-west-2", &abyss.RecordContentDeliveredRequest{
		Data: []*abyss.ContentDelivered{
			{
				Meta: &abyss.AbyssEventMeta{
					Time: now,
				},
				ContentLength: thirtySeconds,
				Stream: &abyss.StreamResource{
					ChannelArn: channelARN,
					StreamId:   1,
				},
			},
		},
	}))
	require.Equal(t, 5, fakeSQS.SendMessageCallCount)

	require.NoError(t, producer.RecordContentRecordedBatch(ctx, "us-west-2", &abyss.RecordContentRecordedRequest{
		Data: []*abyss.ContentRecorded{
			{
				Meta: &abyss.AbyssEventMeta{
					Time: now,
				},
				Duration: thirtySeconds,
				Stream: &abyss.StreamResource{
					ChannelArn: channelARN,
				},
			},
		},
	}))
	require.Equal(t, 6, fakeSQS.SendMessageCallCount)

	require.NoError(t, producer.RecordConcurrentViewsBatch(ctx, "us-west-2", &abyss.RecordConcurrentViewsRequest{
		Data: []*abyss.ConcurrentViews{
			{
				Meta: &abyss.AbyssEventMeta{
					Time: now,
				},
				NumViews: int64(1),
				Stream: &abyss.StreamResource{
					ChannelArn: channelARN,
				},
			},
		},
	}))
	require.Equal(t, 7, fakeSQS.SendMessageCallCount)

	require.NoError(t, producer.RecordConcurrentStreamsBatch(ctx, "us-west-2", &abyss.RecordConcurrentStreamsRequest{
		Data: []*abyss.ConcurrentStreams{
			{
				Meta: &abyss.AbyssEventMeta{
					Time: now,
				},
				TotalCount: int64(1),
				AccountId:  "1234567890",
			},
		},
	}))
	require.Equal(t, 8, fakeSQS.SendMessageCallCount)

	require.NoError(t, producer.RecordMessageReceivedBatch(ctx, "us-west-2", &abyss.RecordMessageReceivedRequest{
		Data: []*abyss.MessageReceived{
			{
				Meta: &abyss.AbyssEventMeta{
					Time: now,
				},
				Room: &abyss.RoomResource{
					Arn: roomARN,
				},
				Type: abyss.MessageType_ROOM_MESSAGE,
				Sender: &abyss.ChatClient{
					Id: "testid",
					Ip: "127.0.0.1",
				},
				MessageLength: 10,
				PayloadBytes:  10,
			},
		},
	}))
	require.Equal(t, 9, fakeSQS.SendMessageCallCount)

	require.NoError(t, producer.RecordMessageDeliveredBatch(ctx, "us-west-2", &abyss.RecordMessageDeliveredRequest{
		Data: []*abyss.MessageDelivered{
			{
				Meta: &abyss.AbyssEventMeta{
					Time: now,
				},
				Room: &abyss.RoomResource{
					Arn: roomARN,
				},
				Type:      abyss.MessageType_ROOM_MESSAGE,
				MessageId: "testmessageid",
				Sender: &abyss.ChatClient{
					Id: "testid",
					Ip: "127.0.0.1",
				},
				Receiver: &abyss.ChatClient{
					Id: "testid",
					Ip: "127.0.0.1",
				},
				MessageLength: 10,
				PayloadBytes:  10,
			},
		},
	}))
	require.Equal(t, 10, fakeSQS.SendMessageCallCount)

	require.NoError(t, producer.RecordConcurrentChatConnectionsBatch(ctx, "us-west-2", &abyss.RecordConcurrentChatConnectionsRequest{
		Data: []*abyss.ConcurrentChatConnections{
			{
				Meta: &abyss.AbyssEventMeta{
					Time: now,
				},
				Room: &abyss.RoomResource{
					Arn: roomARN,
				},
				NumConnections: 10,
			},
		},
	}))
	require.Equal(t, 11, fakeSQS.SendMessageCallCount)
}

type fakeSQS struct {
	SendMessageCallCount int
}

func (sqs *fakeSQS) SendMessageWithContext(aws.Context, *sqs.SendMessageInput, ...request.Option) (*sqs.SendMessageOutput, error) {
	sqs.SendMessageCallCount += 1
	return nil, nil
}

func TestValidateHomeRegion(t *testing.T) {

	testcase := func(homeRegion string, channelARN string, expectedErr bool) func(*testing.T) {
		return func(t *testing.T) {
			regionFromChannel, err := HomeRegionForChannel(channelARN)
			if err != nil {
				t.Fatalf("Parsing chanenl returned error, (err=%v)", err)
				return
			}
			regionLog := fmt.Sprintf("channelFromRegion=%s", regionFromChannel)
			regionLog += fmt.Sprintf("homeRegion=%s", homeRegion)
			if expectedErr {
				if regionFromChannel == homeRegion {
					t.Fatalf("expected an error, but homeRegion == chanelFromRegion %s", regionLog)
					return
				}
			} else {
				if regionFromChannel != homeRegion {
					t.Fatalf("Not expected any error but homeRegion != regionFromChannel, %s", regionLog)
					return
				}
			}
		}
	}
	t.Run("Same home region", testcase("us-west-2", "arn:aws:starfruit:us-west-2:123456789012:channel/whatever", false))
	t.Run("no region", testcase("us-west-2", "arn:aws:starfruit::123456789012:channel/whatever", true))
	t.Run("Different home region", testcase("us-east-2", "arn:aws:starfruit:us-west-2:123456789012:channel/whatever", true))
}

func TestValidateContentTranscoded(t *testing.T) {
	fakeSQS := &fakeSQS{}
	producer, err := NewAbyssProducer("beta", "us-west-2", fakeSQS)
	require.NoError(t, err)

	testcase := func(errMsg string, msg *abyss.ContentTranscoded) func(*testing.T) {
		return func(t *testing.T) {
			err := producer.RecordContentTranscoded(context.Background(), msg)
			require.Error(t, err, "expected an error")
			assert.Contains(t, err.Error(), errMsg, fmt.Sprintf("expected error message to contain %s", errMsg))
		}
	}

	t.Run("missing timestamp", testcase("invalid timestamp", &abyss.ContentTranscoded{
		Meta:   &abyss.AbyssEventMeta{},
		Length: thirtySeconds,
		Stream: &abyss.ContributionStream{
			Stream: &abyss.StreamResource{
				ChannelArn: channelARN,
				StreamId:   1,
			},
		},
	}))

	t.Run("zero timestamp", testcase("invalid timestamp", &abyss.ContentTranscoded{
		Meta: &abyss.AbyssEventMeta{
			Time: zeroTime,
		},
		Length: thirtySeconds,
		Stream: &abyss.ContributionStream{
			Stream: &abyss.StreamResource{
				ChannelArn: channelARN,
				StreamId:   1,
			},
		},
	}))

	t.Run("missing duration", testcase("invalid duration", &abyss.ContentTranscoded{
		Meta: &abyss.AbyssEventMeta{
			Time: now,
		},
		Stream: &abyss.ContributionStream{
			Stream: &abyss.StreamResource{
				ChannelArn: channelARN,
				StreamId:   1,
			},
		},
	}))

	t.Run("missing ARN", testcase("invalid channel ARN", &abyss.ContentTranscoded{
		Meta: &abyss.AbyssEventMeta{
			Time: now,
		},
		Length: thirtySeconds,
		Stream: &abyss.ContributionStream{
			Stream: &abyss.StreamResource{},
		},
	}))

	t.Run("segment missing duration", testcase("segment 1 has invalid duration", &abyss.ContentTranscoded{
		Meta: &abyss.AbyssEventMeta{
			Time: now,
		},
		Length: thirtySeconds,
		Stream: &abyss.ContributionStream{
			Stream: &abyss.StreamResource{
				ChannelArn: channelARN,
				StreamId:   1,
			},
		},
		MediaSegments: []*abyss.MediaSegment{
			{
				SequenceNumber: 1,
			},
		},
	}))

	t.Run("missing stream ID", testcase(ErrInvalidStreamID.Error(), &abyss.ContentTranscoded{
		Meta: &abyss.AbyssEventMeta{
			Time: now,
		},
		Length: thirtySeconds,
		Stream: &abyss.ContributionStream{
			Stream: &abyss.StreamResource{
				ChannelArn: channelARN,
			},
		},
	}))
}

func TestValidateContentContributed(t *testing.T) {
	fakeSQS := &fakeSQS{}
	producer, err := NewAbyssProducer("beta", "us-west-2", fakeSQS)
	require.NoError(t, err)

	testcase := func(errMsg string, msg *abyss.ContentContributed) func(*testing.T) {
		return func(t *testing.T) {
			err := producer.RecordContentContributed(context.Background(), msg)
			require.Error(t, err, "expected an error")
			assert.Contains(t, err.Error(), errMsg, fmt.Sprintf("expected error message to contain %s", errMsg))
		}
	}

	t.Run("missing timestamp", testcase("invalid timestamp", &abyss.ContentContributed{
		Meta:   &abyss.AbyssEventMeta{},
		Length: thirtySeconds,
		Stream: &abyss.ContributionStream{
			Stream: &abyss.StreamResource{
				ChannelArn: channelARN,
			},
		},
	}))

	t.Run("zero timestamp", testcase("invalid timestamp", &abyss.ContentContributed{
		Meta: &abyss.AbyssEventMeta{
			Time: zeroTime,
		},
		Length: thirtySeconds,
		Stream: &abyss.ContributionStream{
			Stream: &abyss.StreamResource{
				ChannelArn: channelARN,
			},
		},
	}))

	t.Run("missing duration", testcase("invalid duration", &abyss.ContentContributed{
		Meta: &abyss.AbyssEventMeta{
			Time: now,
		},
		Stream: &abyss.ContributionStream{
			Stream: &abyss.StreamResource{
				ChannelArn: channelARN,
			},
		},
	}))

	t.Run("missing ARN", testcase("invalid channel ARN", &abyss.ContentContributed{
		Meta: &abyss.AbyssEventMeta{
			Time: now,
		},
		Length: thirtySeconds,
		Stream: &abyss.ContributionStream{
			Stream: &abyss.StreamResource{},
		},
	}))
}

func TestValidateContentDelivered(t *testing.T) {
	fakeSQS := &fakeSQS{}
	producer, err := NewAbyssProducer("beta", "us-west-2", fakeSQS)
	require.NoError(t, err)

	testcase := func(errMsg string, msg *abyss.ContentDelivered) func(*testing.T) {
		return func(t *testing.T) {
			err := producer.RecordContentDelivered(context.Background(), msg)
			require.Error(t, err, "expected an error")
			assert.Contains(t, err.Error(), errMsg, fmt.Sprintf("expected error message to contain %s", errMsg))
		}
	}

	t.Run("missing timestamp", testcase("invalid timestamp", &abyss.ContentDelivered{
		Meta:          &abyss.AbyssEventMeta{},
		ContentLength: thirtySeconds,
		Stream: &abyss.StreamResource{
			ChannelArn: channelARN,
		},
	}))

	t.Run("zero timestamp", testcase("invalid timestamp", &abyss.ContentDelivered{
		Meta: &abyss.AbyssEventMeta{
			Time: zeroTime,
		},
		ContentLength: thirtySeconds,
		Stream: &abyss.StreamResource{
			ChannelArn: channelARN,
		},
	}))

	t.Run("missing duration", testcase("invalid duration", &abyss.ContentDelivered{
		Meta: &abyss.AbyssEventMeta{
			Time: now,
		},
		Stream: &abyss.StreamResource{
			ChannelArn: channelARN,
		},
	}))

	t.Run("missing ARN", testcase("invalid channel ARN", &abyss.ContentDelivered{
		Meta: &abyss.AbyssEventMeta{
			Time: now,
		},
		ContentLength: thirtySeconds,
		Stream:        &abyss.StreamResource{},
	}))
	t.Run("missing stream ID", testcase(ErrInvalidStreamID.Error(), &abyss.ContentDelivered{
		Meta: &abyss.AbyssEventMeta{
			Time: now,
		},
		ContentLength: thirtySeconds,
		Stream: &abyss.StreamResource{
			ChannelArn: channelARN},
	}))
}

func TestValidateContentRecorded(t *testing.T) {
	fakeSQS := &fakeSQS{}
	producer, err := NewAbyssProducer("beta", "us-west-2", fakeSQS)
	require.NoError(t, err)

	testcase := func(errMsg string, msg *abyss.ContentRecorded) func(*testing.T) {
		return func(t *testing.T) {
			err := producer.RecordContentRecorded(context.Background(), msg)
			require.Error(t, err, "expected an error")
			assert.Contains(t, err.Error(), errMsg, fmt.Sprintf("expected error message to contain %s", errMsg))
		}
	}

	t.Run("missing timestamp", testcase("invalid timestamp", &abyss.ContentRecorded{
		Meta:     &abyss.AbyssEventMeta{},
		Duration: thirtySeconds,
		Stream: &abyss.StreamResource{
			ChannelArn: channelARN,
		},
	}))

	t.Run("zero timestamp", testcase("invalid timestamp", &abyss.ContentRecorded{
		Meta: &abyss.AbyssEventMeta{
			Time: zeroTime,
		},
		Duration: thirtySeconds,
		Stream: &abyss.StreamResource{
			ChannelArn: channelARN,
		},
	}))

	t.Run("missing duration", testcase("invalid duration", &abyss.ContentRecorded{
		Meta: &abyss.AbyssEventMeta{
			Time: now,
		},
		Stream: &abyss.StreamResource{
			ChannelArn: channelARN,
		},
	}))

	t.Run("missing ARN", testcase("invalid channel ARN", &abyss.ContentRecorded{
		Meta: &abyss.AbyssEventMeta{
			Time: now,
		},
		Duration: thirtySeconds,
		Stream:   &abyss.StreamResource{},
	}))
}

func TestValidateContentCCV(t *testing.T) {
	t.Run("missing timestamp", func(t *testing.T) {
		err := validateContentCCV(testHomeRegion, []*abyss.ConcurrentViews{
			&abyss.ConcurrentViews{
				Meta:     &abyss.AbyssEventMeta{},
				NumViews: int64(1),
				Stream: &abyss.StreamResource{
					ChannelArn: channelARN,
				},
			},
		})

		require.Error(t, err)
	})

	t.Run("invalid timestamp", func(t *testing.T) {
		err := validateContentCCV(testHomeRegion, []*abyss.ConcurrentViews{
			&abyss.ConcurrentViews{
				Meta: &abyss.AbyssEventMeta{
					Time: zeroTime,
				},
				NumViews: int64(1),
				Stream: &abyss.StreamResource{
					ChannelArn: channelARN,
				},
			},
		})

		require.Error(t, err)
	})

	t.Run("missing identifying information", func(t *testing.T) {
		err := validateContentCCV(testHomeRegion, []*abyss.ConcurrentViews{
			&abyss.ConcurrentViews{
				Meta: &abyss.AbyssEventMeta{
					Time: now,
				},
				NumViews: int64(1),
			},
		})

		require.Error(t, err)
	})

	t.Run("missing stream channel arn", func(t *testing.T) {
		err := validateContentCCV(testHomeRegion, []*abyss.ConcurrentViews{
			&abyss.ConcurrentViews{
				Meta: &abyss.AbyssEventMeta{
					Time: now,
				},
				Stream:   &abyss.StreamResource{},
				NumViews: int64(1),
			},
		})

		require.Error(t, err)
	})

	t.Run("client data region mismatch", func(t *testing.T) {
		err := validateContentCCV(testHomeRegion, []*abyss.ConcurrentViews{
			&abyss.ConcurrentViews{
				Meta: &abyss.AbyssEventMeta{
					Time: now,
				},
				NumViews: int64(1),
				Stream: &abyss.StreamResource{
					ChannelArn: (arn.ARN{
						Region: "us-east-1",
					}).String(),
				},
			},
		})

		require.Error(t, err)
	})

	t.Run("per channel datapoint", func(t *testing.T) {
		err := validateContentCCV(testHomeRegion, []*abyss.ConcurrentViews{
			&abyss.ConcurrentViews{
				Meta: &abyss.AbyssEventMeta{
					Time: now,
				},
				NumViews: int64(1),
				Stream: &abyss.StreamResource{
					ChannelArn: channelARN,
				},
			},
		})

		require.NoError(t, err)
	})

	t.Run("per customer datapoint", func(t *testing.T) {
		err := validateContentCCV(testHomeRegion, []*abyss.ConcurrentViews{
			&abyss.ConcurrentViews{
				Meta: &abyss.AbyssEventMeta{
					Time: now,
				},
				NumViews:  int64(1),
				AccountId: "11111111111",
			},
		})

		require.NoError(t, err)
	})
}

func TestValidateContentCCS(t *testing.T) {
	t.Run("missing timestamp", func(t *testing.T) {
		err := validateContentCCS([]*abyss.ConcurrentStreams{
			&abyss.ConcurrentStreams{
				Meta:       &abyss.AbyssEventMeta{},
				TotalCount: int64(1),
				AccountId:  "1234567890",
			},
		})

		require.Error(t, err)
	})

	t.Run("invalid timestamp", func(t *testing.T) {
		err := validateContentCCS([]*abyss.ConcurrentStreams{
			&abyss.ConcurrentStreams{
				Meta: &abyss.AbyssEventMeta{
					Time: zeroTime,
				},
				TotalCount: int64(1),
				AccountId:  "1234567890",
			},
		})

		require.Error(t, err)
	})

	t.Run("missing account id", func(t *testing.T) {
		err := validateContentCCS([]*abyss.ConcurrentStreams{
			&abyss.ConcurrentStreams{
				Meta: &abyss.AbyssEventMeta{
					Time: now,
				},
				TotalCount: int64(1),
			},
		})

		require.Error(t, err)
	})

	t.Run("happy case", func(t *testing.T) {
		err := validateContentCCS([]*abyss.ConcurrentStreams{
			&abyss.ConcurrentStreams{
				Meta: &abyss.AbyssEventMeta{
					Time: now,
				},
				TotalCount: int64(1),
				AccountId:  "1234567890",
			},
		})

		require.NoError(t, err)
	})
}

func TestValidateMessageReceived(t *testing.T) {
	testCases := []struct {
		desc   string
		data   []*abyss.MessageReceived
		errMsg string
	}{
		{
			desc: "valid messages",
			data: []*abyss.MessageReceived{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					Room: &abyss.RoomResource{
						Arn: roomARN,
					},
					Type: abyss.MessageType_ROOM_MESSAGE,
					Sender: &abyss.ChatClient{
						Id: "testid",
						Ip: "testip",
					},
					Moderated:     false,
					MessageLength: 10,
					PayloadBytes:  10,
				},
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					Room: &abyss.RoomResource{
						Arn: roomARN,
					},
					Type:          abyss.MessageType_SYSTEM_MESSAGE,
					Moderated:     false,
					MessageLength: 10,
					PayloadBytes:  10,
				},
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					Room: &abyss.RoomResource{
						Arn: roomARN,
					},
					Type:          abyss.MessageType_EVENT_MESSAGE,
					Moderated:     false,
					MessageLength: 10,
					PayloadBytes:  10,
				},
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					Room: &abyss.RoomResource{
						Arn: roomARN,
					},
					Type: abyss.MessageType_DISCONNECT_USER,
					Sender: &abyss.ChatClient{
						Id: "testid",
						Ip: "testip",
					},
					TargetId: "testtargetid",
				},
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					Room: &abyss.RoomResource{
						Arn: roomARN,
					},
					Type: abyss.MessageType_DELETE_MESSAGE,
					Sender: &abyss.ChatClient{
						Id: "testid",
						Ip: "testip",
					},
					TargetId: "testtargetid",
				},
			},
		},
		{
			desc: "invalid timestamp",
			data: []*abyss.MessageReceived{
				{
					Meta: &abyss.AbyssEventMeta{},
				},
			},
			errMsg: "invalid timestamp",
		},
		{
			desc: "zero timestamp",
			data: []*abyss.MessageReceived{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: zeroTime,
					},
				},
			},
			errMsg: "invalid timestamp",
		},
		{
			desc: "invalid room",
			data: []*abyss.MessageReceived{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
				},
			},
			errMsg: "invalid room",
		},
		{
			desc: "invalid room arn",
			data: []*abyss.MessageReceived{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					Room: &abyss.RoomResource{
						Arn: "foobar",
					},
				},
			},
			errMsg: "invalid room ARN",
		},
		{
			desc: "invalid message type",
			data: []*abyss.MessageReceived{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					Room: &abyss.RoomResource{
						Arn: roomARN,
					},
					Type: -1,
				},
			},
			errMsg: "invalid message type",
		},
	}
	for _, tC := range testCases {
		t.Run(tC.desc, func(t *testing.T) {
			err := validateMessageReceived(tC.data)
			if tC.errMsg != "" {
				assert.Contains(t, err.Error(), tC.errMsg)
			} else {
				assert.NoError(t, err)
			}
		})
	}
}

func TestValidateRoomMessageReceived(t *testing.T) {
	testCases := []struct {
		desc   string
		data   *abyss.MessageReceived
		errMsg string
	}{
		{
			desc:   "room message nil sender",
			data:   &abyss.MessageReceived{},
			errMsg: "sender is nil",
		},
		{
			desc: "room message invalid sender id",
			data: &abyss.MessageReceived{
				Sender: &abyss.ChatClient{},
			},
			errMsg: "invalid sender ID",
		},
		{
			desc: "room message invalid sender ip",
			data: &abyss.MessageReceived{
				Sender: &abyss.ChatClient{
					Id: "testid",
				},
			},
			errMsg: "invalid sender IP",
		}, {
			desc: "room message invalid sender ip",
			data: &abyss.MessageReceived{
				Sender: &abyss.ChatClient{
					Id: "testid",
				},
			},
			errMsg: "invalid sender IP",
		},
	}
	for _, tC := range testCases {
		t.Run(tC.desc, func(t *testing.T) {
			err := validateRoomMessageReceived(tC.data)
			if tC.errMsg != "" {
				assert.Contains(t, err.Error(), tC.errMsg)
			} else {
				assert.NoError(t, err)
			}
		})
	}
}

func TestValidateOperationReceived(t *testing.T) {
	testCases := []struct {
		desc   string
		data   *abyss.MessageReceived
		errMsg string
	}{
		{
			desc: "success",
			data: &abyss.MessageReceived{
				Sender: &abyss.ChatClient{
					Id: "testid",
					Ip: "127.0.0.1",
				},
				TargetId: "testtargetid",
			},
		}, {
			desc:   "operation nil sender",
			data:   &abyss.MessageReceived{},
			errMsg: "sender is nil",
		},
		{
			desc: "operation invalid sender id",
			data: &abyss.MessageReceived{
				Sender: &abyss.ChatClient{},
			},
			errMsg: "invalid sender ID",
		},
		{
			desc: "operation invalid sender ip",
			data: &abyss.MessageReceived{
				Sender: &abyss.ChatClient{
					Id: "testid",
				},
			},
			errMsg: "invalid sender IP",
		},
		{
			desc: "operation invalid sender ip",
			data: &abyss.MessageReceived{
				Sender: &abyss.ChatClient{
					Id: "testid",
				},
			},
			errMsg: "invalid sender IP",
		},
		{
			desc: "operation missing target id",
			data: &abyss.MessageReceived{
				Sender: &abyss.ChatClient{
					Id: "testid",
					Ip: "127.0.0.1",
				},
			},
			errMsg: "invalid target ID",
		},
	}

	for _, tC := range testCases {
		err := validateOperationReceived(tC.data)
		if tC.errMsg != "" {
			assert.Contains(t, err.Error(), tC.errMsg)
		} else {
			assert.NoError(t, err)
		}
	}
}

func TestValidateRoomMessageDelivered(t *testing.T) {
	testCases := []struct {
		desc   string
		data   *abyss.MessageDelivered
		errMsg string
	}{
		{
			desc: "success",
			data: &abyss.MessageDelivered{
				Meta: &abyss.AbyssEventMeta{
					Time: now,
				},
				Type: abyss.MessageType_ROOM_MESSAGE,
				Room: &abyss.RoomResource{
					Arn: roomARN,
				},
				Sender: &abyss.ChatClient{
					Id: "testid",
					Ip: "testip",
				},
				MessageLength: 10,
				PayloadBytes:  10,
			},
		},
		{
			desc:   "room message nil sender",
			data:   &abyss.MessageDelivered{},
			errMsg: "sender is nil",
		},
		{
			desc: "room message invalid sender id",
			data: &abyss.MessageDelivered{
				Sender: &abyss.ChatClient{},
			},
			errMsg: "invalid sender ID",
		},
		{
			desc: "room message invalid sender ip",
			data: &abyss.MessageDelivered{
				Sender: &abyss.ChatClient{
					Id: "testid",
				},
			},
			errMsg: "invalid sender IP",
		},
		{
			desc: "room message invalid sender ip",
			data: &abyss.MessageDelivered{
				Sender: &abyss.ChatClient{
					Id: "testid",
				},
			},
			errMsg: "invalid sender IP",
		},
	}
	for _, tC := range testCases {
		t.Run(tC.desc, func(t *testing.T) {
			err := validateRoomMessageDelivered(tC.data)
			if tC.errMsg != "" {
				assert.Contains(t, err.Error(), tC.errMsg)
			} else {
				assert.NoError(t, err)
			}
		})
	}
}

func TestValidateMessageDelivered(t *testing.T) {
	testCases := []struct {
		desc   string
		data   []*abyss.MessageDelivered
		errMsg string
	}{
		{
			desc: "success",
			data: []*abyss.MessageDelivered{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					MessageId: "testmessageid",
					Room: &abyss.RoomResource{
						Arn: roomARN,
					},
					Type: abyss.MessageType_ROOM_MESSAGE,
					Sender: &abyss.ChatClient{
						Id: "testid",
						Ip: "testip",
					},
					Receiver: &abyss.ChatClient{
						Id: "testid",
						Ip: "testip",
					},
					MessageLength: 10,
					PayloadBytes:  10,
				},
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					MessageId: "testmessageid",
					Room: &abyss.RoomResource{
						Arn: roomARN,
					},
					Type: abyss.MessageType_SYSTEM_MESSAGE,
					Receiver: &abyss.ChatClient{
						Id: "testid",
						Ip: "testip",
					},
					MessageLength: 10,
					PayloadBytes:  10,
				},
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					MessageId: "testmessageid",
					Room: &abyss.RoomResource{
						Arn: roomARN,
					},
					Type: abyss.MessageType_EVENT_MESSAGE,
					Receiver: &abyss.ChatClient{
						Id: "testid",
						Ip: "testip",
					},
					MessageLength: 10,
					PayloadBytes:  10,
				},
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					MessageId: "testmessageid",
					Room: &abyss.RoomResource{
						Arn: roomARN,
					},
					Type: abyss.MessageType_DISCONNECT_USER,
					Sender: &abyss.ChatClient{
						Id: "testid",
						Ip: "testip",
					},
					Receiver: &abyss.ChatClient{
						Id: "testid",
						Ip: "testip",
					},
				},
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					MessageId: "testmessageid",
					Room: &abyss.RoomResource{
						Arn: roomARN,
					},
					Type: abyss.MessageType_DELETE_MESSAGE,
					Sender: &abyss.ChatClient{
						Id: "testid",
						Ip: "testip",
					},
					Receiver: &abyss.ChatClient{
						Id: "testid",
						Ip: "testip",
					},
				},
			},
		},
		{
			desc: "invalid timestamp",
			data: []*abyss.MessageDelivered{
				{
					Meta: &abyss.AbyssEventMeta{},
				},
			},
			errMsg: "invalid timestamp",
		},
		{
			desc: "zero timestamp",
			data: []*abyss.MessageDelivered{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: zeroTime,
					},
				},
			},
			errMsg: "invalid timestamp",
		},
		{
			desc: "invalid room",
			data: []*abyss.MessageDelivered{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
				},
			},
			errMsg: "invalid room",
		},
		{
			desc: "invalid room arn",
			data: []*abyss.MessageDelivered{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					Room: &abyss.RoomResource{
						Arn: "foobar",
					},
				},
			},
			errMsg: "invalid room ARN",
		},
		{
			desc: "invalid message type",
			data: []*abyss.MessageDelivered{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					Room: &abyss.RoomResource{
						Arn: roomARN,
					},
					MessageId: "testmessageid",
					Receiver: &abyss.ChatClient{
						Id: "testreceiverid",
						Ip: "127.0.0.1",
					},
					Type: -1,
				},
			},
			errMsg: "invalid message type",
		},
		{
			desc: "missing message id",
			data: []*abyss.MessageDelivered{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					Room: &abyss.RoomResource{
						Arn: roomARN,
					},
				},
			},
			errMsg: "invalid MessageId",
		},
		{
			desc: "missing receiver",
			data: []*abyss.MessageDelivered{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					Room: &abyss.RoomResource{
						Arn: roomARN,
					},
					MessageId: "testmessageid",
				},
			},
			errMsg: "invalid receiver",
		},
		{
			desc: "missing receiver id",
			data: []*abyss.MessageDelivered{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					Room: &abyss.RoomResource{
						Arn: roomARN,
					},
					MessageId: "testmessageid",
					Receiver:  &abyss.ChatClient{},
				},
			},
			errMsg: "invalid receiver",
		},
		{
			desc: "missing receiver ip",
			data: []*abyss.MessageDelivered{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					Room: &abyss.RoomResource{
						Arn: roomARN,
					},
					MessageId: "testmessageid",
					Receiver: &abyss.ChatClient{
						Id: "testreceiverid",
					},
				},
			},
			errMsg: "invalid receiver",
		},
	}
	for _, tC := range testCases {
		t.Run(tC.desc, func(t *testing.T) {
			err := validateMessageDelivered(tC.data)
			if tC.errMsg != "" {
				assert.Contains(t, err.Error(), tC.errMsg)
			} else {
				assert.NoError(t, err)
			}
		})
	}
}

func TestValidateChatCCC(t *testing.T) {
	testCases := []struct {
		desc   string
		data   []*abyss.ConcurrentChatConnections
		errMsg string
	}{
		{
			desc: "success",
			data: []*abyss.ConcurrentChatConnections{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					Room: &abyss.RoomResource{
						Arn: roomARN,
					},
				},
			},
		},
		{
			desc: "invalid timestamp",
			data: []*abyss.ConcurrentChatConnections{
				{
					Meta: &abyss.AbyssEventMeta{},
				},
			},
			errMsg: "invalid timestamp",
		},
		{
			desc: "zero timestamp",
			data: []*abyss.ConcurrentChatConnections{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: zeroTime,
					},
				},
			},
			errMsg: "invalid timestamp",
		},
		{
			desc: "invalid room",
			data: []*abyss.ConcurrentChatConnections{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
				},
			},
			errMsg: "invalid room",
		},
		{
			desc: "invalid room arn",
			data: []*abyss.ConcurrentChatConnections{
				{
					Meta: &abyss.AbyssEventMeta{
						Time: now,
					},
					Room: &abyss.RoomResource{
						Arn: "foobar",
					},
				},
			},
			errMsg: "invalid room ARN",
		},
	}
	for _, tC := range testCases {
		t.Run(tC.desc, func(t *testing.T) {
			err := validateChatCCC(tC.data)
			if tC.errMsg != "" {
				assert.Contains(t, err.Error(), tC.errMsg)
			} else {
				assert.NoError(t, err)
			}
		})
	}
}
