package StarfruitAbyssProducer

import (
	"fmt"
	"os"

	"github.com/aws/aws-sdk-go/aws/arn"
)

type (
	messageType     string
	queueRegionType string
)

const (
	deliver          messageType = "deliver"
	transcode        messageType = "transcode"
	record           messageType = "record"
	contribute       messageType = "contribute"
	ccs              messageType = "ccs"
	ccv              messageType = "ccv"
	messageReceived  messageType = "messagereceived"
	messageDelivered messageType = "messagedelivered"
	ccc              messageType = "ccc"

	boundRegionType queueRegionType = "bound"
	homeRegionType  queueRegionType = "home"
)

var (
	homeRegions         = []string{"us-west-2", "us-east-1", "eu-west-1"}
	boundRegions        = []string{"us-west-2", "us-east-2", "eu-west-2"}
	homeRegionMessages  = []messageType{deliver, ccs, ccv, messageReceived, messageDelivered, ccc}
	boundRegionMessages = []messageType{transcode, record, contribute}

	// Map of stage -> queue region -> ID of account hosting SQS queues
	accountIDs = map[string]map[string]string{
		"prod": {
			"us-west-2": "500600498289",
			"us-east-2": "422782966314",
			"eu-west-2": "308327167373",
			"us-east-1": "609420647978",
			"eu-west-1": "428871488439",
		},
		"beta": {
			"us-west-2": "717518721197",
			"us-east-2": "118171646437",
			"eu-west-2": "978151196077",
			"us-east-1": "353792302699",
			"eu-west-1": "012203525622",
		},
		"dev": {
			"us-west-2": os.Getenv("AWS_ACCOUNT_ID"),
			"us-east-2": os.Getenv("AWS_ACCOUNT_ID"),
			"eu-west-2": os.Getenv("AWS_ACCOUNT_ID"),
			"us-east-1": os.Getenv("AWS_ACCOUNT_ID"),
			"eu-west-1": os.Getenv("AWS_ACCOUNT_ID"),
		},
		"local": {
			"us-west-2": os.Getenv("SAM_ASSUME_ACCOUNT_ID"),
			"us-east-2": os.Getenv("SAM_ASSUME_ACCOUNT_ID"),
			"eu-west-2": os.Getenv("SAM_ASSUME_ACCOUNT_ID"),
			"us-east-1": os.Getenv("SAM_ASSUME_ACCOUNT_ID"),
			"eu-west-1": os.Getenv("SAM_ASSUME_ACCOUNT_ID"),
		},
	}

	messagesByQueueRegion = make(map[messageType]queueRegionType)
	regionIsBound         = make(map[string]bool)
	regionIsHome          = make(map[string]bool)
)

func init() {
	for _, message := range boundRegionMessages {
		messagesByQueueRegion[message] = boundRegionType
	}
	for _, message := range homeRegionMessages {
		messagesByQueueRegion[message] = homeRegionType
	}

	for _, homeRegion := range homeRegions {
		regionIsHome[homeRegion] = true
	}

	for _, boundRegion := range boundRegions {
		regionIsBound[boundRegion] = true
	}
}

func queueAccount(stage, producerRegion string) (string, error) {
	stageMap, ok := accountIDs[stage]
	if !ok {
		return "", fmt.Errorf("stage %q has no queues", stage)
	}

	accountID, ok := stageMap[producerRegion]
	if !ok {
		return "", fmt.Errorf("queues are not set up for %q stage in %q region", stage, producerRegion)
	}

	return accountID, nil
}

func queueURL(stage, producerRegion, homeRegion string, message messageType) (string, error) {
	queueRegion, ok := messagesByQueueRegion[message]
	if !ok {
		return "", fmt.Errorf("no queue region for %q", message)
	}

	switch queueRegion {
	case homeRegionType:
		if !regionIsHome[producerRegion] {
			isBoundRegion := regionIsBound[producerRegion]
			regionDescription := "unknown"
			if isBoundRegion {
				regionDescription = "bound"
			}

			return "", fmt.Errorf("%q must be used in a home region. %q is a %s region.", message, producerRegion, regionDescription)
		}
	case boundRegionType:
		if !regionIsBound[producerRegion] {
			isBoundRegion := regionIsHome[producerRegion]
			regionDescription := "unknown"
			if isBoundRegion {
				regionDescription = "home"
			}

			return "", fmt.Errorf("%q must be used in a bound region. %q is a %s region.", message, producerRegion, regionDescription)
		}
	}

	accountID, err := queueAccount(stage, producerRegion)
	if err != nil {
		return "", err
	}

	queueName := fmt.Sprintf("%s-region-%s-%s", queueRegion, message, homeRegion)
	url := fmt.Sprintf("https://sqs.%s.amazonaws.com/%s/%s", producerRegion, accountID, queueName)
	return url, nil
}

// HomeRegionForChannel returns the home region ID where the channel's data
// should be sent (for example "us-west-2"). If the channel ARN is not parsable,
// this may return an error.
func HomeRegionForChannel(channelARN string) (string, error) {
	parsed, err := arn.Parse(channelARN)
	if err != nil {
		// note that %w is not being used because gotranscoder is not using go1.13
		// https://git-aws.internal.justin.tv/video/gotranscoder/blob/master/bingo.toml
		return "", fmt.Errorf("failed to parse arn %q: %s", channelARN, err)
	}
	return parsed.Region, nil
}
