package StarfruitAbyssProducer

import (
	"strings"
	"testing"

	"github.com/stretchr/testify/require"
)

func TestHomeRegionForChannel(t *testing.T) {
	testcase := func(channelARN, want string, wantErr bool) func(*testing.T) {
		return func(t *testing.T) {
			have, err := HomeRegionForChannel(channelARN)
			if wantErr {
				if err == nil {
					t.Fatalf("expected an error, but its nil (got region=%q)", have)
				}

				if channelARN != "" && !strings.Contains(err.Error(), channelARN) {
					t.Fatalf("expected error to have channel arn but didn't: %q", channelARN)
				}
				return
			}

			if err != nil {
				t.Fatalf("unexpected err: %v", err)
			}

			if have != want {
				t.Error("wrong region")
				t.Logf("have=%q", have)
				t.Logf("want=%q", want)
			}
		}
	}

	t.Run("empty", testcase("", "", true))
	t.Run("not an arn", testcase("lirik", "", true))
	t.Run("no region", testcase("arn:aws:starfruit::123456789012:channel/whatever", "", false))
	t.Run("valid arn", testcase("arn:aws:starfruit:us-west-2:123456789012:channel/whatever", "us-west-2", false))
}

func TestQueueURL(t *testing.T) {
	for _, tt := range []struct {
		name           string
		stage          string
		producerRegion string
		homeRegion     string
		messageType    messageType

		wantURL string
		errLike string
	}{
		{
			name:           "valid prod bound",
			stage:          "prod",
			producerRegion: "us-west-2", homeRegion: "us-east-1", messageType: contribute,
			wantURL: "https://sqs.us-west-2.amazonaws.com/500600498289/bound-region-contribute-us-east-1",
		},
		{
			name:           "valid prod home",
			stage:          "prod",
			producerRegion: "eu-west-1", homeRegion: "eu-west-1", messageType: deliver,
			wantURL: "https://sqs.eu-west-1.amazonaws.com/428871488439/home-region-deliver-eu-west-1",
		},
		{
			name:           "valid prod home messageReceived",
			stage:          "prod",
			producerRegion: "eu-west-1", homeRegion: "eu-west-1", messageType: messageReceived,
			wantURL: "https://sqs.eu-west-1.amazonaws.com/428871488439/home-region-messagereceived-eu-west-1",
		},
		{
			name:           "valid prod home messageDelivered",
			stage:          "prod",
			producerRegion: "eu-west-1", homeRegion: "eu-west-1", messageType: messageDelivered,
			wantURL: "https://sqs.eu-west-1.amazonaws.com/428871488439/home-region-messagedelivered-eu-west-1",
		},
		{
			name:           "valid prod home ccc",
			stage:          "prod",
			producerRegion: "eu-west-1", homeRegion: "eu-west-1", messageType: ccc,
			wantURL: "https://sqs.eu-west-1.amazonaws.com/428871488439/home-region-ccc-eu-west-1",
		},
		{
			name:           "valid prod us-west-2 home",
			stage:          "prod",
			producerRegion: "us-west-2",
			homeRegion:     "us-west-2",
			messageType:    deliver,
			wantURL:        "https://sqs.us-west-2.amazonaws.com/500600498289/home-region-deliver-us-west-2",
		},
		{
			name:           "valid prod us-west-2 bound",
			stage:          "prod",
			producerRegion: "us-west-2",
			homeRegion:     "us-west-2",
			messageType:    contribute,
			wantURL:        "https://sqs.us-west-2.amazonaws.com/500600498289/bound-region-contribute-us-west-2",
		},
		{
			name:           "invalid prod home",
			stage:          "prod",
			producerRegion: "us-east-1",
			homeRegion:     "us-east-1",
			messageType:    contribute,

			errLike: `"contribute" must be used in a bound region. "us-east-1" is a home region.`,
		},
		{
			name:           "invalid prod bound",
			stage:          "prod",
			producerRegion: "us-east-2",
			homeRegion:     "us-east-1",
			messageType:    deliver,

			errLike: `"deliver" must be used in a home region. "us-east-2" is a bound region.`,
		},
		{
			name:           "valid beta bound",
			stage:          "beta",
			producerRegion: "us-east-2",
			homeRegion:     "us-east-1",
			messageType:    contribute,
			wantURL:        "https://sqs.us-east-2.amazonaws.com/118171646437/bound-region-contribute-us-east-1",
		},
		{
			name:           "invalid beta home",
			stage:          "beta",
			producerRegion: "us-east-1",
			homeRegion:     "us-east-1",
			messageType:    contribute,
			errLike:        `"contribute" must be used in a bound region. "us-east-1" is a home region.`,
		},
		{
			name:           "valid beta home",
			stage:          "beta",
			producerRegion: "eu-west-1",
			homeRegion:     "eu-west-1",
			messageType:    deliver,
			wantURL:        "https://sqs.eu-west-1.amazonaws.com/012203525622/home-region-deliver-eu-west-1",
		},
		{
			name:           "invalid beta bound",
			stage:          "beta",
			producerRegion: "us-east-2",
			homeRegion:     "us-east-1",
			messageType:    deliver,
			errLike:        `"deliver" must be used in a home region. "us-east-2" is a bound region.`,
		},
		{
			name:           "invalid stage",
			stage:          "staging",
			producerRegion: "us-east-2",
			homeRegion:     "us-east-1",
			messageType:    contribute,
			errLike:        `stage "staging" has no queues`,
		},
		{
			name:           "invalid bound region",
			stage:          "prod",
			producerRegion: "potato",
			homeRegion:     "us-east-1",
			messageType:    contribute,
			errLike:        `"contribute" must be used in a bound region. "potato" is a unknown region.`,
		},
		{
			name:           "invalid bound region",
			stage:          "prod",
			producerRegion: "potato",
			homeRegion:     "us-east-1",
			messageType:    contribute,
			errLike:        `"contribute" must be used in a bound region. "potato" is a unknown region.`,
		},
		{
			name:           "invalid bound region messageReceived",
			stage:          "prod",
			producerRegion: "us-east-2",
			homeRegion:     "us-east-1",
			messageType:    messageReceived,
			errLike:        `"messagereceived" must be used in a home region. "us-east-2" is a bound region.`,
		},
		{
			name:           "invalid bound region messageDelivered",
			stage:          "prod",
			producerRegion: "us-east-2",
			homeRegion:     "us-east-1",
			messageType:    messageDelivered,
			errLike:        `"messagedelivered" must be used in a home region. "us-east-2" is a bound region.`,
		},
		{
			name:           "invalid bound region ccc",
			stage:          "prod",
			producerRegion: "us-east-2",
			homeRegion:     "us-east-1",
			messageType:    ccc,
			errLike:        `"ccc" must be used in a home region. "us-east-2" is a bound region.`,
		},
	} {
		t.Run(tt.name, func(t *testing.T) {
			haveURL, err := queueURL(tt.stage, tt.producerRegion, tt.homeRegion, tt.messageType)
			if tt.errLike != "" {
				require.Error(t, err)

				require.Contains(t, err.Error(), tt.errLike)

				return
			}
			require.NoError(t, err)

			require.Equal(t, tt.wantURL, haveURL)
		})
	}
}
