package consumer

import (
	"context"
	"fmt"
	"sort"
	"testing"
	"time"

	serviceapi "code.justin.tv/amzn/StarfruitMOLLELambdaTwirp"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/client"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sns/snsiface"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/aws/aws-sdk-go/service/sqs/sqsiface"
)

// newTestClient constructs a MOLLEConsumer that can be called without a
// valid AWS session. It won't do anything useful and probably just
// hang trying to receive messages.
func newTestClient() *molleConsumer {
	return &molleConsumer{
		consumers: []envConsumer{
			{
				sqs: &testSQSClient{},
			},
		},
	}
}

type testSQSClient struct {
	sqsiface.SQSAPI
	region string
}

type testSNSClient struct {
	snsiface.SNSAPI
}

// GetQueueUrl successfully returns a realistic value
func (c *testSQSClient) GetQueueUrl(input *sqs.GetQueueUrlInput) (*sqs.GetQueueUrlOutput, error) {
	return c.GetQueueUrlWithContext(context.Background(), input)
}

// GetQueueUrlWithContext successfully returns a realistic value
func (c *testSQSClient) GetQueueUrlWithContext(ctx context.Context, input *sqs.GetQueueUrlInput, opts ...request.Option) (*sqs.GetQueueUrlOutput, error) {
	return &sqs.GetQueueUrlOutput{
		QueueUrl: aws.String(fmt.Sprintf("https://sqs.%v.amazonaws.com/%v/%v", c.region, aws.StringValue(input.QueueOwnerAWSAccountId), aws.StringValue(input.QueueName))),
	}, nil
}

// ReceiveMessagesWithContext blocks until the context is canceled.
func (c *testSQSClient) ReceiveMessageWithContext(ctx aws.Context, input *sqs.ReceiveMessageInput, opts ...request.Option) (*sqs.ReceiveMessageOutput, error) {
	<-ctx.Done()
	return nil, ctx.Err()
}

// DeleteMessageBatchWithContext returns an empty response.
func (c *testSQSClient) DeleteMessageBatchWithContext(ctx aws.Context, input *sqs.DeleteMessageBatchInput, opts ...request.Option) (*sqs.DeleteMessageBatchOutput, error) {
	return &sqs.DeleteMessageBatchOutput{}, nil
}

// TestContextCancel checks that the background goroutines spawned by
// molleConsumer.StartFetchAndHandle promptly terminate, signalled by
// the returned output channel being closed, when the provided context
// is canceled.
func TestContextCancel(t *testing.T) {
	client := newTestClient()

	handle := func(serviceapi.WeaverEvent, time.Time) error { return nil }

	ctx, cancel := context.WithCancel(context.Background())
	c := client.StartFetchAndHandle(ctx, handle, nil)
	cancel()

	//

	select {
	case <-time.After(200 * time.Millisecond):
		t.Error("timeout waiting for client goroutines to terminate")
	case _, ok := <-c:
		if ok {
			t.Error("received unexpected value from output channel")
		}
	}
}

func TestWithEnvs(t *testing.T) {
	cfg := []EnvConfig{
		{"333333333333", "us-east-2", "molle-stage-3", "-suffix-3"},
		{"444444444444", "us-west-2", "molle-stage-4", "-suffix-4"},
	}

	makeClients := func(s client.ConfigProvider, cfgs ...*aws.Config) (sqsiface.SQSAPI, snsiface.SNSAPI) {
		var region string
		for _, cfg := range cfgs {
			if cfg.Region != nil {
				region = *cfg.Region
			}
		}

		return &testSQSClient{region: region}, &testSNSClient{}
	}

	ctx := context.Background()

	consumerClient, err := NewConsumerClientWithMollEEnvsAndClients(ctx, "test", "consumer-env-2", cfg, session.New(), makeClients)
	if err != nil {
		t.Fatalf("couldn't create consumer client: %v", err)
	}

	consumers := consumerClient.(*molleConsumer).consumers
	sort.Slice(consumers, func(i, j int) bool { return consumers[i].snsTopicARN < consumers[j].snsTopicARN })

	if len(consumers) != 2 {
		t.Fatalf("expected 2 consumers, got %v", len(consumers))
	}

	expectedQueueURL := "https://sqs.us-east-2.amazonaws.com/333333333333/video-weaver-consumer-env-2-test-suffix-3"
	if consumers[0].sqsQueueURL != expectedQueueURL {
		t.Fatalf("expected queue url %v, got %v", expectedQueueURL, consumers[0].sqsQueueURL)
	}

	expectedQueueURL = "https://sqs.us-west-2.amazonaws.com/444444444444/video-weaver-consumer-env-2-test-suffix-4"
	if consumers[1].sqsQueueURL != expectedQueueURL {
		t.Fatalf("expected queue url %v, got %v", expectedQueueURL, consumers[1].sqsQueueURL)
	}

	expectedTopicARN := "arn:aws:sns:us-east-2:333333333333:WeaverErrors-molle-stage-3"
	if consumers[0].snsTopicARN != expectedTopicARN {
		t.Fatalf("expected topic arn %v, got %v", expectedTopicARN, consumers[0].snsTopicARN)
	}
	expectedTopicARN = "arn:aws:sns:us-west-2:444444444444:WeaverErrors-molle-stage-4"
	if consumers[1].snsTopicARN != expectedTopicARN {
		t.Fatalf("expected topic arn %v, got %v", expectedTopicARN, consumers[1].snsTopicARN)
	}
}
