package consumer

import (
	"context"
	"time"

	"github.com/cenkalti/backoff"
	"github.com/pkg/errors"
)

// backoffWaiter provides exponential backoff.
type backoffWaiter struct {
	backoff *backoff.ExponentialBackOff
}

func newBackoffWaiter() *backoffWaiter {
	expBackoff := backoff.NewExponentialBackOff()
	expBackoff.RandomizationFactor = 0
	expBackoff.MaxElapsedTime = 0
	expBackoff.MaxInterval = 20 * time.Second

	return &backoffWaiter{
		backoff: expBackoff,
	}
}

// Wait will block until ctx is Done or for the current backoff period,
// which is also exponentially increased.
func (b *backoffWaiter) Wait(ctx context.Context) {
	t := time.NewTimer(b.backoff.NextBackOff())
	defer t.Stop()

	select {
	case <-ctx.Done():
	case <-t.C:
	}
}

// Reset resets the backoff period to the initial value.
func (b *backoffWaiter) Reset() {
	b.backoff.Reset()
}

// nonRetryableError is a marker type around an error.
type nonRetryableError struct {
	error
}

// isNonRetryable checks whether there is a nonRetryableError in the
// error's Cause chain.
func isNonRetryable(err error) bool {
	var placeholder nonRetryableError
	return errors.As(err, &placeholder)
}

// Cause returns the wrapped error.
func (err nonRetryableError) Cause() error {
	return err.error
}

// NonRetryable wraps the error in a marker type that suggests that the
// failed operation should not be retried. Returning such a wrapper
// error from the StartFetchAndHandle handler function will cause the
// SQS message to be deleted so it won't be received by us again after
// the visibility timeout.
func NonRetryable(err error) error {
	return nonRetryableError{error: err}
}
