package StarfruitNyxClient

import (
	"context"
	"math/rand"
	"time"

	rpc "code.justin.tv/amzn/StarfruitNyxTwirp"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/video/invoker"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/kinesis"
	"github.com/aws/aws-sdk-go/service/kinesis/kinesisiface"
	"github.com/cenkalti/backoff"
	"github.com/klauspost/compress/zstd"
)

type streamConfig struct {
	region           Region
	eventType        EventType
	awsSession       *session.Session
	awsCreds         *credentials.Credentials
	sampleReporter   *telemetry.SampleReporter
	encoder          *zstd.Encoder
	onError          func(region string, err error)
	flushInterval    time.Duration
	sendTimeout      time.Duration
	sendRetryTimeout time.Duration
}

type stream struct {
	name           string
	stream         string
	eventTypeName  string
	client         kinesisiface.KinesisAPI
	sampleReporter *telemetry.SampleReporter

	events           chan ProtoMessage
	encoder          *zstd.Encoder
	payload          *rpc.NyxPayload
	uncompressedData []byte
	compressedData   []byte
	eventCount       int
	esres            *rpc.EdgeSegmentRequestEvents
	epres            *rpc.EdgePlaylistRequestEvents
	rsres            *rpc.ReplicationSegmentRequestEvents
	beacons          *rpc.BeaconEvents

	onError          func(region string, err error)
	timer            *time.Timer
	flushInterval    time.Duration
	rand             *rand.Rand
	sendTimeout      time.Duration
	sendRetryTimeout time.Duration

	inv *invoker.Tasks
}

func newStream(c *streamConfig) (*stream, error) {
	s := &stream{
		name:          c.region.name,
		stream:        c.eventType.stream,
		eventTypeName: c.eventType.name,
		client: kinesis.New(c.awsSession, &aws.Config{
			Credentials: c.awsCreds,
			Region:      aws.String(c.region.amazonName),
			MaxRetries:  aws.Int(0),
		}),
		sampleReporter: c.sampleReporter,
		events:         make(chan ProtoMessage, eventsChannelSize),
		encoder:        c.encoder,
		payload: &rpc.NyxPayload{
			PayloadType: c.eventType.payloadType,
		},
		uncompressedData: make([]byte, 0, eventDataSize),
		compressedData:   make([]byte, 0, eventDataSize),
		esres: &rpc.EdgeSegmentRequestEvents{
			Events: make([]*rpc.EdgeSegmentRequestEvent, 1),
		},
		epres: &rpc.EdgePlaylistRequestEvents{
			Events: make([]*rpc.EdgePlaylistRequestEvent, 1),
		},
		rsres: &rpc.ReplicationSegmentRequestEvents{
			Events: make([]*rpc.ReplicationSegmentRequestEvent, 1),
		},
		beacons: &rpc.BeaconEvents{
			Events: make([]*rpc.BeaconEvent, 1),
		},
		onError:          c.onError,
		flushInterval:    c.flushInterval,
		rand:             rand.New(rand.NewSource(time.Now().UnixNano())),
		sendTimeout:      30 * time.Second,
		sendRetryTimeout: time.Minute,
		inv:              invoker.New(),
	}

	if c.sendTimeout != 0 {
		s.sendTimeout = c.sendTimeout
	}

	if c.sendRetryTimeout != 0 {
		s.sendRetryTimeout = c.sendRetryTimeout
	}

	return s, nil
}

// run loop
func (s *stream) run(baseCtx context.Context) error {
	s.inv.Add(func(ctx context.Context) error {
		for {
			var timerCh <-chan time.Time
			if s.timer != nil {
				timerCh = s.timer.C
			}

			select {
			case <-timerCh:
				s.flush()
			case e := <-s.events:
				s.addEventToBatch(e)
			case <-ctx.Done(): // Server is shutting down
				// This is race-y but we don't know how many connections are in-flight (hopefully 0)
				// but we already gave up on them so also give up on their events
				numEvents := len(s.events)
				for i := 0; i < numEvents; i++ {
					s.addEventToBatch(<-s.events)
				}
				s.flush()
				return nil
			}
		}
	})

	return s.inv.Run(baseCtx)
}

// event addition
func (s *stream) addEvent(event ProtoMessage) error {
	select {
	case s.events <- event:
		return nil
	default:
		if s.sampleReporter != nil {
			s.sampleReporter.Report("Nyx"+s.eventTypeName+"QueueFull"+s.name, 1.0, telemetry.UnitCount)
		}
		return ErrEventQueueFull
	}
}

func (s *stream) addEventBlocking(ctx context.Context, event ProtoMessage) error {
	select {
	case s.events <- event:
		return nil
	case <-ctx.Done():
		if s.sampleReporter != nil {
			s.sampleReporter.Report("Nyx"+s.eventTypeName+"QueueCancelled"+s.name, 1.0, telemetry.UnitCount)
		}
		return ErrEventQueueFull
	}
}

func (s *stream) addEventToBatch(e ProtoMessage) {
	var batch ProtoMessage
	switch ev := e.(type) {
	case *rpc.EdgeSegmentRequestEvent:
		s.esres.Events[0] = ev
		batch = s.esres
	case *rpc.EdgePlaylistRequestEvent:
		s.epres.Events[0] = ev
		batch = s.epres
	case *rpc.ReplicationSegmentRequestEvent:
		s.rsres.Events[0] = ev
		batch = s.rsres
	case *rpc.BeaconEvent:
		s.beacons.Events[0] = ev
		batch = s.beacons
	default:
		panic("unknown event type")
	}

	size := batch.Size()
	if size+len(s.uncompressedData) > cap(s.uncompressedData) {
		s.flush()
	}

	n, err := batch.MarshalTo(s.uncompressedData[len(s.uncompressedData) : len(s.uncompressedData)+size])

	if err != nil {
		if s.onError != nil {
			s.onError(s.name, ErrMessageMarshalFailed)
		}
		return
	}

	s.uncompressedData = s.uncompressedData[:len(s.uncompressedData)+n]
	s.eventCount++

	if s.timer == nil && s.flushInterval != 0 {
		s.timer = time.NewTimer(s.flushInterval)
	}
}

// event sends
func (s *stream) flush() {
	// we're going to choose to ignore the ctx and always attempt to finish the flush
	if len(s.uncompressedData) == 0 {
		return
	}

	if s.timer != nil {
		s.timer.Stop()
		s.timer = nil
	}

	// marshal data
	s.compressedData = s.encoder.EncodeAll(s.uncompressedData, s.compressedData)

	if len(s.compressedData) < len(s.uncompressedData) {
		s.payload.CompressionLevel = rpc.CompressionLevel_ZSTD
		s.payload.Data = s.compressedData
	} else {
		s.payload.CompressionLevel = rpc.CompressionLevel_UNCOMPRESSED
		s.payload.Data = s.uncompressedData
	}

	body, _ := s.payload.Marshal() // This can not error, so ignore it
	s.uncompressedData = s.uncompressedData[:0]
	s.compressedData = s.compressedData[:0]

	eventCount := s.eventCount
	s.eventCount = 0

	key := make([]byte, 64)
	s.rand.Read(key)

	s.inv.Add(s.send(body, key, eventCount))
}

func (s *stream) send(body []byte, key []byte, eventCount int) invoker.Task {
	return func(ctx context.Context) error {
		// send data

		// maximum retry delays under this config:
		// 1s, 1.65s, 2.72s, 4.5s, 5.5s, 5.5s, ...
		retryer := backoff.NewExponentialBackOff()
		retryer.InitialInterval = time.Second
		retryer.RandomizationFactor = 0.1
		retryer.MaxInterval = 5 * time.Second
		retryer.MaxElapsedTime = s.sendRetryTimeout

		start := time.Now()
		retryCount := 0.0
		err := backoff.RetryNotify(func() error {
			kctx, cancel := context.WithTimeout(context.Background(), s.sendTimeout)
			defer cancel()

			attemptStartTime := time.Now()
			_, err := s.client.PutRecordWithContext(kctx, &kinesis.PutRecordInput{
				StreamName:   aws.String(s.stream),
				PartitionKey: aws.String(string(key)),
				Data:         body,
			})
			attemptDuration := time.Since(attemptStartTime)

			if s.sampleReporter != nil {
				if err != nil {
					s.sampleReporter.Report("NyxPut"+s.eventTypeName+"Failed"+s.name, 1.0, telemetry.UnitCount)
					s.sampleReporter.Report("NyxPut"+s.eventTypeName+"FailedSize"+s.name, float64(eventCount), telemetry.UnitCount)
					s.sampleReporter.ReportDurationSample("NyxPut"+s.eventTypeName+"FailedDuration"+s.name, attemptDuration)
				} else {
					s.sampleReporter.Report("NyxPut"+s.eventTypeName+"Succeeded"+s.name, 1.0, telemetry.UnitCount)
					s.sampleReporter.Report("NyxPut"+s.eventTypeName+"SucceededSize"+s.name, float64(eventCount), telemetry.UnitCount)
					s.sampleReporter.ReportDurationSample("NyxPut"+s.eventTypeName+"SucceededDuration"+s.name, attemptDuration)
				}
			}

			if err != nil && s.onError != nil {
				s.onError(s.name, err)
			}

			return err
		}, retryer, func(err error, t time.Duration) {
			retryCount++
		})

		if s.sampleReporter != nil {
			s.sampleReporter.ReportDurationSample("NyxPut"+s.eventTypeName+"TotalDuration"+s.name, time.Since(start))

			if err != nil {
				s.sampleReporter.Report("NyxPut"+s.eventTypeName+"Dropped"+s.name, 1.0, telemetry.UnitCount)
				s.sampleReporter.Report("NyxPut"+s.eventTypeName+"DroppedSize"+s.name, float64(eventCount), telemetry.UnitCount)
			} else {
				s.sampleReporter.Report("NyxPut"+s.eventTypeName+"SucceededRetries"+s.name, retryCount, telemetry.UnitCount)
			}
		}

		return nil
	}
}
