package StarfruitSECProducer

import (
	"fmt"
	"strings"
)

const (
	channelPrefixStarfruit = "aws."
	channelPrefixLVS       = "lvs."
)

// GetChannelArn returns the channel ARN given a channel name
// This is a shim for now. Eventually this would be modified when we are close to SF GA.
func GetChannelArn(channel string) string {
	if IsStarfruitChannel(channel) {
		arn := strings.Replace(channel, "channel.", "channel/", -1)
		return "arn:" + strings.Replace(arn, ".", ":", -1)
	}
	// the LVS check handles both LVS and SVS channels
	if IsLvsChannel(channel) {
		customerId, contentId := GetCustomerIdContentIdFromChannel(channel)
		return fmt.Sprintf("arn:aws:lvs:us-west-2:%s:channel/%s", customerId, contentId)
	}
	return fmt.Sprintf("arn:aws:twitch:us-west-2:twitch:channel/%s", channel)
}

// IsStarfruitChannel returns if the given channel is an Starfruit channel or not
func IsStarfruitChannel(channel string) bool {
	prefixed := isChannelPrefixed(channel, channelPrefixStarfruit)
	return prefixed && getChannelPartsCount(channel) == 6
}

// IsLvsChannel returns if the given channel is an LVS / SVS channel or not
func IsLvsChannel(channel string) bool {
	prefixed := isChannelPrefixed(channel, channelPrefixLVS)
	return prefixed && getChannelPartsCount(channel) == 3
}

// isChannelPrefixed returns if a channel is prefixed with the provided prefix string
func isChannelPrefixed(channel string, prefix string) bool {
	if channel == "" {
		return false
	}
	if !strings.HasPrefix(channel, prefix) {
		return false
	}
	return true
}

func getChannelPartsCount(channel string) int {
	data := strings.Split(channel, ".")
	return len(data)
}

// GetCustomerIdContentIdFromChannel will get the customer id and content id from channel
func GetCustomerIdContentIdFromChannel(channel string) (string, string) {
	// Check for both LVS and SVS channels
	if IsStarfruitChannel(channel) {
		data := strings.Split(channel, ".")
		return data[3], data[5]
	} else if IsLvsChannel(channel) {
		data := strings.Split(channel, ".")
		return data[1], data[2]
	}
	return "", ""
}
