package StarfruitSECProducer

import (
	"bufio"
	"fmt"
	"log"
	"os"
	"strings"
	"testing"
	"time"

	"code.justin.tv/amzn/StarfruitSECTwirp"
	"github.com/golang/protobuf/proto"
)

// By default this test is skipped since it is not a unit test.
// The purpose of this test is evaluate and determine optimal circular buffer size
// given different parameters like incoming message rate, latency and buffer size.
// Ref. - https://docs.google.com/document/d/1bdGtrIdwdIjprlDfCtwqW9bq16b5XveGScq4YWB6O2o/edit?usp=sharing
// To run the test, set environmental variable RunCirBufLoadTest to true. When it is not defined or false,
// the test is skipped.
// On MacOS zsh:
// > export RunCirBufLoadTest=true
// with execute the test.
// Locally the test result file is created in build output dir, like for example:
// .../workspaces/StarfruitSEC/build/StarfruitSECProducer/StarfruitSECProducer-1.0/AL2_x86_64/DEV.STD.PTHREAD/build/gopath/src/code.justin.tv/amzn/StarfruitSECProducer

type LoadTestRun struct {
	resultWriter  *bufio.Writer
	bufferSizes   []int
	latencies     []float64
	incomingRates []int
}

type LoadTest struct {
	bufferSize   int
	latency      time.Duration
	incomingRate int
	testDuration time.Duration
	msgReceived  int
	msgLost      int
}

func NewLoadTestRun() *LoadTestRun {
	l := LoadTestRun{}
	fileName := fmt.Sprintf("CircularBufferLoadTestRun-%d.csv", time.Now().Nanosecond())
	f, err := os.Create(fileName)
	if err != nil {
		log.Fatal(err)
	}
	l.resultWriter = bufio.NewWriter(f)
	return &l
}

func (ltr *LoadTestRun) RunLoadTest(bss []int, lts []float64, irs []int, td int) {
	// argument passed is a value in seconds
	testDuration := time.Duration(td) * time.Second
	for _, bs := range bss {
		for _, lt := range lts {
			// latency, argument is float seconds
			ltns := time.Duration(lt) * time.Second
			for _, ir := range irs {
				nlt := NewLoadTest(bs, ltns, ir, testDuration)
				nlt.runTest()
				ltr.writeTestResult(nlt)
			}
		}
	}
	ltr.resultWriter.Flush()
}

func (ltr *LoadTestRun) writeTestResult(nlt *LoadTest) {
	_, err := ltr.resultWriter.WriteString(nlt.toString())
	if err != nil {
		log.Fatal(err)
	}
}

func NewLoadTest(bs int, latency time.Duration, rate int, dur time.Duration) *LoadTest {
	l := LoadTest{
		bufferSize:   bs,
		latency:      latency,
		incomingRate: rate,
		testDuration: dur,
	}
	return &l
}

func (lt *LoadTest) runTest() {
	lt.msgReceived, lt.msgLost = GenerateLoad(lt.bufferSize, lt.latency, lt.incomingRate, lt.testDuration)
}

func (l *LoadTest) toString() string {
	return fmt.Sprintf("%d,%9.3f,%d,%d,%d,%d\n", l.bufferSize, float64(l.latency)/1000000000.0, l.incomingRate, l.testDuration, l.msgReceived, l.msgLost)
}

var counter int

func NewTestPayload() proto.Message {
	counter++
	msg := StarfruitSECTwirp.StreamEventData{
		//Extra: fmt.Sprintf("stream details json - %d", time.Now().Unix()),

		Extra: fmt.Sprintf("stream details json - %d", counter),
	}
	batchMsg := &StarfruitSECTwirp.StreamEventDataBatch{
		Data: []*StarfruitSECTwirp.StreamEventData{&msg},
	}
	return batchMsg
}

// GenerateLoad
// BufferSize - size of circular buffer in terms of how many messages are held
// latency - response latency we simulate for SQS response, in nanoseconds
// incomingRate - how messages per second
// testDuration - how long we want to run the simulation, in seconds
// returned value - messages lost
func GenerateLoad(bufferSize int, latencyInNanoSec time.Duration, incomingRate int, testDurationInSec time.Duration) (int, int) {
	// Setup Test
	mockSvc := &MockSQSClient{
		delay: latencyInNanoSec,
	}
	// msg loss MsgLossDuration = 60 seconds i.e. we compute message loss per minute
	cfg := Config{
		SqsApi:          mockSvc,
		QueueURL:        "QueueURL",
		BufferSize:      bufferSize,
		MsgLossDuration: DefaultMsgLossDuration,
		// no logging call back set
	}
	psp := createProtoSQSProducer(cfg)
	start := time.Now()
	sec := 0
	msgReceived := 0
	for time.Since(start) < testDurationInSec {
		fmt.Printf("%f < %f \n", time.Since(start).Seconds(), float64(testDurationInSec))
		msgReceived = msgReceived + sendTestEvents(&psp, incomingRate)
		sec++
		fmt.Printf("Seconds passed %d\n", sec)
	}
	fmt.Printf("Done sending events\n")
	return msgReceived, psp.chnBuf.msgLoss()
}

func sendTestEvents(p ProtobufSQSProducer, incomingRate int) int {
	start := time.Now()
	i := 0
	for i < incomingRate {
		p.Send(nil, NewTestPayload())
		i++
	}
	// we finished in less than 1 second, so we need to sleep for the remaining time
	time.Sleep(time.Until(start.Add(time.Second)))
	fmt.Printf("Sent %d messages\n", i)
	return i
}

func TestLoadGeneration(t *testing.T) {
	if !strings.EqualFold(os.Getenv("RunCirBufLoadTest"), "true") {
		t.Skip("skipping Channel Buffer Load Test")
	}
	fmt.Println("**************************************************")
	fmt.Println("GenerateLoad starts")
	fmt.Println("**************************************************")
	//bs := [6]int{2, 4, 8, 10}
	//lts := [5]float64{0.001, 0.01, 0.1, 1.0, 10.0}
	//irs := [5]int{1, 10, 20, 50, 100}
	bs := [3]int{80, 100, 120}
	lts := [3]float64{0.01, 0.1, 1.0}
	irs := [4]int{100, 125, 150, 200}
	ltr := NewLoadTestRun()
	ltr.RunLoadTest(bs[0:], lts[0:], irs[0:], 1)
	fmt.Println("**************************************************")
	fmt.Println("GenerateLoad DONE")
	fmt.Println("**************************************************")

}
