package StarfruitSECProducer

import (
	"context"
	"encoding/base64"
	"fmt"
	"sync"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/aws/aws-sdk-go/service/sqs/sqsiface"
	"github.com/golang/protobuf/proto"
	"github.com/pkg/errors"
)

const (
	// MaxPayloadSize is the maximum permitted size in bytes of a message (measured
	// with the MessageSize function) that can be send in SQS.
	// Limit is 262,144 bytes:		https://docs.aws.amazon.com/sdk-for-go/api/service/sqs/#SQS.SendMessage
	MaxPayloadSize = 600 * 400

	// DefaultMsgBufferSize Buffer size
	DefaultMsgBufferSize = 100

	// DefaultMsgLossDuration Default period over which message loss is computed.
	// We default to 1 minute at present.
	DefaultMsgLossDuration = time.Duration(time.Minute)
)

// MessageSize returns the size of a message if it were to be encoded into a
// payload for transmission in SQS. Valid messages should not exceed
// MaxPayloadSize.
func MessageSize(msg proto.Message) int {
	// proto.Size is cheap in modern editions of generated code from
	// protoc-gen-go, since it produces a XXX_Size() int method on all generated
	// objects which gets called, so this doens't require fully marshaling the
	// message.
	return base64.StdEncoding.EncodedLen(proto.Size(msg))
}

// ErrorMessageTooBig is an error indicating that a message cannot be sent to
// SQS because it is too large when encoded.
type ErrorMessageTooBig struct {
	// Size is the encoded size of the message.
	Size int
	// Message is the message that was too large to encode.
	Message proto.Message
}

func (e ErrorMessageTooBig) Error() string {
	return fmt.Sprintf("message of length %d bytes exceeds maximum sendable size", e.Size)
}

type ProtobufSQSProducer interface {
	Send(ctx context.Context, payload proto.Message) error
}

// messageWithContext We define a structure which clubs together message input and the context in which it is sent.
type messageWithContext struct {
	msgInput *sqs.SendMessageInput
	ctx      context.Context
}

// A ProtoSQSProducer encodes proto messages for transport in SQS, and sends
// them on a specific queue.
//
// This is a low-level client. Most users will want to use SECProducer.
type ProtoSQSProducer struct {
	sqs             sqsiface.SQSAPI
	queueURL        string
	chnBuf          *channelBuffer
	msgLossDuration time.Duration // in seconds
	loggingCallBack func(s string)
	mux             sync.Mutex // we use while setting & accessing logging call back
}

type Config struct {

	// QueueURL - url for SQS from where SEC reads events
	QueueURL string

	// SqsApi - SQS API facade
	SqsApi sqsiface.SQSAPI

	// BufferSize - how many messages can be hold in underlying buffer
	BufferSize int

	// MsgLossDuration - this is the MsgLossDuration over which message loss is counted
	//				after this period, the message loss is set back to 0
	//				this way we can know rate of message loss, if it occurs
	MsgLossDuration time.Duration

	// LoggingCallBack - Call back provided by the calling service to log
	LoggingCallBack func(s string)
}

// NewProtoSQSProducerByConfig Creates new SEC event producer/writer, for public consumption.
func NewProtoSQSProducerByConfig(sqspCfg Config) ProtobufSQSProducer {
	producer := createProtoSQSProducer(sqspCfg)
	return &producer
}

// createProtoSQSProducer - concretely produces the underlying implementation structure.
//                          internal method for unit testing purposes (TODO - mock)
func createProtoSQSProducer(sqspCfg Config) ProtoSQSProducer {
	producer := ProtoSQSProducer{
		queueURL:        sqspCfg.QueueURL,
		sqs:             sqspCfg.SqsApi,
		msgLossDuration: sqspCfg.MsgLossDuration,
		loggingCallBack: sqspCfg.LoggingCallBack,
	}
	producer.chnBuf = newChannelBuffer(sqspCfg.BufferSize, sqspCfg.MsgLossDuration, sqspCfg.LoggingCallBack)
	// If we pass producer argument to the go func, subsequent setter calls
	// to set logger do not get reflect the update logger.
	// sWe use closure here. Ref. - https://golangbot.com/first-class-functions/
	go producer.consumeMessageBuffer()
	return producer
}

func NewProtoSQSProducer(qurl string, sapi sqsiface.SQSAPI) ProtobufSQSProducer {
	cfg := Config{
		QueueURL:        qurl,
		SqsApi:          sapi,
		BufferSize:      DefaultMsgBufferSize,
		MsgLossDuration: DefaultMsgLossDuration,
	}
	return NewProtoSQSProducerByConfig(cfg)
}

func (p *ProtoSQSProducer) encode(msg proto.Message) (*sqs.SendMessageInput, error) {
	size := MessageSize(msg)
	if size > MaxPayloadSize {
		return nil, ErrorMessageTooBig{Size: size, Message: msg}
	}
	b, err := proto.Marshal(msg)
	if err != nil {
		return nil, err
	}

	body := base64.StdEncoding.EncodeToString(b)
	v := &sqs.SendMessageInput{
		QueueUrl:    aws.String(p.queueURL),
		MessageBody: aws.String(body),
	}

	return v, nil
}

func (p *ProtoSQSProducer) Send(ctx context.Context, payload proto.Message) error {
	req, err := p.encode(payload)
	if err != nil {
		return errors.Wrap(err, "unable to encode message")
	}
	// push the message and context on the circular buffer
	mwc := messageWithContext{
		msgInput: req,
		ctx:      ctx,
	}
	p.chnBuf.push(&mwc)
	return nil
}

func (p *ProtoSQSProducer) consumeMessageBuffer() {
	// infinite loop - waits for a new element to become available when buffer is empty
	// standard for loop struct - init; condition; every iteration execution
	for mwc, more := p.chnBuf.pull(); more; mwc, more = p.chnBuf.pull() {
		if mwc != nil {
			_, err := p.sqs.SendMessageWithContext(mwc.ctx, mwc.msgInput)
			if err != nil {
				f := p.loggingCallBack
				if f != nil {
					f("[SEC-Event-Producer] error sending event to SEC: " + err.Error())
				}
				// we have an error, but since no logging call back is provided
				// we have no choice apart from dropping the error
			}
		}
	}
}

type channelBuffer struct {
	buffer    chan *messageWithContext
	mu        sync.Mutex
	mlTracker *msgLossTracker
}

// s: buffer size
// d: MsgLossDuration over which msg loss is counted
func newChannelBuffer(s int, d time.Duration, lp func(s string)) *channelBuffer {
	f := channelBuffer{
		buffer:    make(chan *messageWithContext, s),
		mlTracker: newTracker(d, lp),
	}
	return &f
}

// returns loss message
func (cb *channelBuffer) push(mwc *messageWithContext) {
	cb.mu.Lock()
	if len(cb.buffer) == cap(cb.buffer) {
		// drop the oldest message
		<-cb.buffer
		// we note the loss
		cb.mlTracker.recordMsgLoss()
	}
	// Mutex ensures that while a capacity is created, in case buffer was full;
	// it is not lost to any other insertion. As a result caller of pus - Send method -
	// does not block.
	cb.buffer <- mwc
	cb.mu.Unlock()
}

func (cb *channelBuffer) pull() (*messageWithContext, bool) {
	result, ok := <-cb.buffer
	return result, ok
}

func (cb *channelBuffer) msgLoss() int {
	return cb.mlTracker.loss()
}

type msgLossTracker struct {
	counter int
	mutex   sync.Mutex
}

// MsgLossDuration over which the message loss is tracked
func newTracker(d time.Duration, lp func(s string)) *msgLossTracker {
	t := msgLossTracker{}
	go func(ld time.Duration, f func(s string)) {
		ticker := time.NewTicker(ld)
		defer ticker.Stop()
		for _ = range ticker.C {
			t.mutex.Lock()
			if t.counter > 0 {
				if f != nil {
					f(fmt.Sprintf("[SEC-Event-Producer] lost %d messages in period %s", t.counter, ld))
				}
			}
			t.counter = 0
			t.mutex.Unlock()
		}
	}(d, lp)
	return &t
}

// records that a message is lost
func (t *msgLossTracker) recordMsgLoss() {
	t.mutex.Lock()
	t.counter++
	t.mutex.Unlock()
}

func (t *msgLossTracker) loss() int {
	t.mutex.Lock()
	defer t.mutex.Unlock()
	return t.counter
}
