package StarfruitSECProducer

import (
	"fmt"

	"github.com/aws/aws-sdk-go/aws/arn"
)

// Map of stage -> home region -> ID of account hosting SQS queues
var accountIDs = map[string]map[string]string{
	"prod": {
		"us-west-2":      "828014892355",
		"us-east-1":      "650802573617",
		"eu-west-1":      "899118124128",
		"ap-northeast-1": "384401017145",
		"ap-northeast-2": "567607597022",
		"ap-south-1":     "011691476354",
		"eu-central-1":   "590388033914",
	},
	"beta": {
		"us-west-2": "401151757587",
	},
	"gamma": {
		"us-west-2": "926040607189",
	},
	"n2t": {
		"us-west-2": "test",
		"us-east-1": "test",
		"eu-west-1": "test",
	},
}

func ValidateProducerParams(stage string, homeRegion string) error {
	stageMap, ok := accountIDs[stage]
	if !ok {
		return fmt.Errorf("stage %q has no queues", stage)
	}

	_, ok = stageMap[homeRegion]
	if !ok {
		return fmt.Errorf("queues are not set up for %q stage in %q region", stage, homeRegion)
	}

	return nil
}

func GetQueueUrl(stage string, homeRegion string) (string, error) {
	err := ValidateProducerParams(stage, homeRegion)
	if err != nil {
		return "", err
	}
	accountID := accountIDs[stage][homeRegion]
	queueName := fmt.Sprintf("StarfruitSEC-%s-%s-StreamEventsQueue", homeRegion, stage)
	url := fmt.Sprintf("https://sqs.%s.amazonaws.com/%s/%s", homeRegion, accountID, queueName)
	if stage == "n2t" {
		url = fmt.Sprintf("http://n2t-sqs:9324/%s/%s", accountID, queueName)
	}
	return url, nil
}

// HomeRegionForChannel returns the home region ID where the channel's data
// should be sent (for example "us-west-2"). If the channel ARN is not parsable,
// this may return an error.
func GetHomeRegionForChannel(channelARN string) (string, error) {
	parsed, err := arn.Parse(channelARN)
	if err != nil {
		return "", err
	}
	return parsed.Region, nil
}
