package StarfruitSECProducer

import "testing"

func TestGetHomeRegionForChannel(t *testing.T) {
	type args struct {
		channelARN string
	}
	tests := []struct {
		name    string
		args    args
		want    string
		wantErr bool
	}{
		{
			name: "TestGetHomeRegionForChannelPDX",
			args: args{
				channelARN: "arn:aws:svs:us-west-2:956482054022:channel:65ce4cbf-fab9-4d4f-9aa1-0843a3256234",
			},
			want:    "us-west-2",
			wantErr: false,
		},
		{
			name: "TestGetHomeRegionForChannelIAD",
			args: args{
				channelARN: "arn:aws:svs:us-east-1:956482054022:channel:65ce4cbf-fab9-4d4f-9aa1-0843a3256234",
			},
			want:    "us-east-1",
			wantErr: false,
		},
		{
			name: "TestGetHomeRegionForChannelDUB",
			args: args{
				channelARN: "arn:aws:svs:eu-west-1:956482054022:channel:65ce4cbf-fab9-4d4f-9aa1-0843a3256234",
			},
			want:    "eu-west-1",
			wantErr: false,
		},
		{
			name: "TestGetHomeRegionForChannelTwitch",
			args: args{
				channelARN: "arn:aws:twitch:us-west-2:twitch:channel:twitch_test",
			},
			want:    "us-west-2",
			wantErr: false,
		},
		{
			name: "TestGetHomeRegionForChannelLVS",
			args: args{
				channelARN: "arn:aws:lvs:us-west-2:amazon-live:channel:65ce4cbf-fab9-4d4f-9aa1-0843a3256234",
			},
			want:    "us-west-2",
			wantErr: false,
		},
		{
			name: "TestGetHomeRegionForChannelTwitch",
			args: args{
				channelARN: "twitch-test",
			},
			want:    "",
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := GetHomeRegionForChannel(tt.args.channelARN)
			if (err != nil) != tt.wantErr {
				t.Errorf("GetHomeRegionForChannel() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if got != tt.want {
				t.Errorf("GetHomeRegionForChannel() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestGetQueueUrl(t *testing.T) {
	type args struct {
		stage      string
		homeRegion string
	}
	tests := []struct {
		name    string
		args    args
		want    string
		wantErr bool
	}{
		{
			name: "TestGetQueueUrlBetaPDX",
			args: args{
				stage:      "beta",
				homeRegion: "us-west-2",
			},
			want:    "https://sqs.us-west-2.amazonaws.com/401151757587/StarfruitSEC-us-west-2-beta-StreamEventsQueue",
			wantErr: false,
		},
		{
			name: "TestGetQueueUrlBetaIAD",
			args: args{
				stage:      "beta",
				homeRegion: "us-east-1",
			},
			want:    "",
			wantErr: true,
		},
		{
			name: "TestGetQueueUrlBetaDUB",
			args: args{
				stage:      "beta",
				homeRegion: "eu-west-1",
			},
			want:    "",
			wantErr: true,
		},
		{
			name: "TestGetQueueUrlGammaPDX",
			args: args{
				stage:      "gamma",
				homeRegion: "us-west-2",
			},
			want:    "https://sqs.us-west-2.amazonaws.com/926040607189/StarfruitSEC-us-west-2-gamma-StreamEventsQueue",
			wantErr: false,
		},
		{
			name: "TestGetQueueUrlGammaIAD",
			args: args{
				stage:      "gamma",
				homeRegion: "us-east-1",
			},
			want:    "",
			wantErr: true,
		},
		{
			name: "TestGetQueueUrlGammaDUB",
			args: args{
				stage:      "gamma",
				homeRegion: "eu-west-1",
			},
			want:    "",
			wantErr: true,
		},
		{
			name: "TestGetQueueUrlProdPDX",
			args: args{
				stage:      "prod",
				homeRegion: "us-west-2",
			},
			want:    "https://sqs.us-west-2.amazonaws.com/828014892355/StarfruitSEC-us-west-2-prod-StreamEventsQueue",
			wantErr: false,
		},
		{
			name: "TestGetQueueUrlProdIAD",
			args: args{
				stage:      "prod",
				homeRegion: "us-east-1",
			},
			want:    "https://sqs.us-east-1.amazonaws.com/650802573617/StarfruitSEC-us-east-1-prod-StreamEventsQueue",
			wantErr: false,
		},
		{
			name: "TestGetQueueUrlProdDUB",
			args: args{
				stage:      "prod",
				homeRegion: "eu-west-1",
			},
			want:    "https://sqs.eu-west-1.amazonaws.com/899118124128/StarfruitSEC-eu-west-1-prod-StreamEventsQueue",
			wantErr: false,
		},
		{
			name: "TestGetQueueUrlN2T",
			args: args{
				stage:      "n2t",
				homeRegion: "us-west-2",
			},
			want:    "http://n2t-sqs:9324/test/StarfruitSEC-us-west-2-n2t-StreamEventsQueue",
			wantErr: false,
		},
		{
			name: "TestGetQueueUrlRandomPDX",
			args: args{
				stage:      "random",
				homeRegion: "us-west-2",
			},
			want:    "",
			wantErr: true,
		},
		{
			name: "TestGetQueueUrlRandomCMH",
			args: args{
				stage:      "random",
				homeRegion: "us-east-2",
			},
			want:    "",
			wantErr: true,
		},
		{
			name: "TestGetQueueUrlRandomIAD",
			args: args{
				stage:      "random",
				homeRegion: "us-east-1",
			},
			want:    "",
			wantErr: true,
		},
		{
			name: "TestGetQueueUrlRandomDUB",
			args: args{
				stage:      "random",
				homeRegion: "eu-west-1",
			},
			want:    "",
			wantErr: true,
		}, {
			name: "TestGetQueueUrlRandomNRT",
			args: args{
				stage:      "random",
				homeRegion: "ap-northeast-1",
			},
			want:    "",
			wantErr: true,
		},
		{
			name: "TestGetQueueUrlRandomICN",
			args: args{
				stage:      "random",
				homeRegion: "ap-northeast-2",
			},
			want:    "",
			wantErr: true,
		},
		{
			name: "TestGetQueueUrlRandomBOM",
			args: args{
				stage:      "random",
				homeRegion: "ap-south-1",
			},
			want:    "",
			wantErr: true,
		},
		{
			name: "TestGetQueueUrlRandomFRA",
			args: args{
				stage:      "random",
				homeRegion: "eu-central-1",
			},
			want:    "",
			wantErr: true,
		},
		{
			name: "TestGetQueueUrlProdNRT",
			args: args{
				stage:      "prod",
				homeRegion: "ap-northeast-1",
			},
			want:    "https://sqs.ap-northeast-1.amazonaws.com/384401017145/StarfruitSEC-ap-northeast-1-prod-StreamEventsQueue",
			wantErr: false,
		},
		{
			name: "TestGetQueueUrlProdICN",
			args: args{
				stage:      "prod",
				homeRegion: "ap-northeast-2",
			},
			want:    "https://sqs.ap-northeast-2.amazonaws.com/567607597022/StarfruitSEC-ap-northeast-2-prod-StreamEventsQueue",
			wantErr: false,
		},
		{
			name: "TestGetQueueUrlProdBOM",
			args: args{
				stage:      "prod",
				homeRegion: "ap-south-1",
			},
			want:    "https://sqs.ap-south-1.amazonaws.com/011691476354/StarfruitSEC-ap-south-1-prod-StreamEventsQueue",
			wantErr: false,
		},
		{
			name: "TestGetQueueUrlProdFRA",
			args: args{
				stage:      "prod",
				homeRegion: "eu-central-1",
			},
			want:    "https://sqs.eu-central-1.amazonaws.com/590388033914/StarfruitSEC-eu-central-1-prod-StreamEventsQueue",
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := GetQueueUrl(tt.args.stage, tt.args.homeRegion)
			if (err != nil) != tt.wantErr {
				t.Errorf("GetQueueUrl() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if got != tt.want {
				t.Errorf("GetQueueUrl() got = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestValidateProducerParams(t *testing.T) {
	type args struct {
		stage      string
		homeRegion string
	}
	tests := []struct {
		name    string
		args    args
		wantErr bool
	}{
		{
			name: "TestValidateProducerParamsValidProdPDX",
			args: args{
				stage:      "prod",
				homeRegion: "us-west-2",
			},
			wantErr: false,
		},
		{
			name: "TestValidateProducerParamsValidProdIAD",
			args: args{
				stage:      "prod",
				homeRegion: "us-east-1",
			},
			wantErr: false,
		},
		{
			name: "TestValidateProducerParamsBetaPDX",
			args: args{
				stage:      "beta",
				homeRegion: "us-west-2",
			},
			wantErr: false,
		},
		{
			name: "TestValidateProducerParamsValidBetaIAD",
			args: args{
				stage:      "beta",
				homeRegion: "us-east-1",
			},
			wantErr: true,
		},
		{
			name: "TestValidateProducerParamsN2TPDX",
			args: args{
				stage:      "n2t",
				homeRegion: "us-west-2",
			},
			wantErr: false,
		},
		{
			name: "TestValidateProducerParamsGammaPDX",
			args: args{
				stage:      "gamma",
				homeRegion: "us-west-2",
			},
			wantErr: false,
		},
		{
			name: "TestValidateProducerParamsGammaIAD",
			args: args{
				stage:      "gamma",
				homeRegion: "us-east-1",
			},
			wantErr: true,
		},
		{
			name: "TestValidateProducerParamsNoStage",
			args: args{
				stage:      "",
				homeRegion: "us-west-2",
			},
			wantErr: true,
		},
		{
			name: "TestValidateProducerParamsNoRegion",
			args: args{
				stage:      "beta",
				homeRegion: "",
			},
			wantErr: true,
		},
		{
			name: "TestValidateProducerParamsWrongStage",
			args: args{
				stage:      "staging",
				homeRegion: "us-west-2",
			},
			wantErr: true,
		},
		{
			name: "TestValidateProducerParamsWrongRegion",
			args: args{
				stage:      "beta",
				homeRegion: "us-west-3",
			},
			wantErr: true,
		},
		{
			name: "TestValidateProducerParamsWrongRegionAndStage",
			args: args{
				stage:      "staging",
				homeRegion: "cmh",
			},
			wantErr: true,
		},
		{
			name: "TestValidateProducerParamsEmpthRegionAndStage",
			args: args{
				stage:      "",
				homeRegion: "",
			},
			wantErr: true,
		},
		{
			name: "TestValidateProducerParamsDUBNoRegion",
			args: args{
				stage:      "",
				homeRegion: "eu-west-1",
			},
			wantErr: true,
		},
		{
			name: "TestValidateProducerParamsDUBAndStage",
			args: args{
				stage:      "prod",
				homeRegion: "eu-west-1",
			},
			wantErr: false,
		},
		{
			name: "TestValidateProducerParamsNRTAndStage",
			args: args{
				stage:      "prod",
				homeRegion: "ap-northeast-1",
			},
			wantErr: false,
		},
		{
			name: "TestValidateProducerParamsICNAndStage",
			args: args{
				stage:      "prod",
				homeRegion: "ap-northeast-2",
			},
			wantErr: false,
		},
		{
			name: "TestValidateProducerParamsBOMAndStage",
			args: args{
				stage:      "prod",
				homeRegion: "ap-south-1",
			},
			wantErr: false,
		},
		{
			name: "TestValidateProducerParamsFRAAndStage",
			args: args{
				stage:      "prod",
				homeRegion: "eu-central-1",
			},
			wantErr: false,
		},
		{
			name: "TestValidateProducerParamsNRTNoRegion",
			args: args{
				stage:      "",
				homeRegion: "ap-northeast-1",
			},
			wantErr: true,
		},
		{
			name: "TestValidateProducerParamsICNANdBadRegion",
			args: args{
				stage:      "staging",
				homeRegion: "ap-northeast-2",
			},
			wantErr: true,
		},
		{
			name: "TestValidateProducerParamsBOMAndBeta",
			args: args{
				stage:      "beta",
				homeRegion: "eu-west-1",
			},
			wantErr: true,
		},
		{
			name: "TestValidateProducerParamsFRAAndGamma",
			args: args{
				stage:      "gamma",
				homeRegion: "eu-west-1",
			},
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if err := ValidateProducerParams(tt.args.stage, tt.args.homeRegion); (err != nil) != tt.wantErr {
				t.Errorf("ValidateProducerParams() error = %v, wantErr %v", err, tt.wantErr)
			}
		})
	}
}
