package sdc

import (
	"context"
	"fmt"
	"log"
	"os"
	"time"

	"code.justin.tv/raymolin/sdc/sdcagent"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
)

const SECProducerConfigPrefix = "SECProducerConfig"

type SECProducerConfig interface {
	// Used for debugging -- each config should be exposed with its own Getter.
	GetSECProducerConfig() *Configs
	GetEnablePublishTranscodeHeartbeat(defaultVal bool) bool
	GetEnablePublishStarvationHeartbeat(defaultVal bool) bool
	GetForcePublishEventsForTwitch(defaultVal bool) bool
	GetForcePublishEventsForLvs(defaultVal bool) bool
	GetEnablePublishEventsToSec(defaultVal bool) bool
	GetTranscodeHeartbeatInterval(defaultVal int32) int32
	GetStarvationHeartbeatInterval(defaultVal int32) int32
}

type Configs struct {
	EnablePublishTranscodeHeartbeat  *bool  `json:"enable-publish-transcode-heartbeat,omitempty"`
	EnablePublishStarvationHeartbeat *bool  `json:"enable-publish-starvation-heartbeat,omitempty"`
	ForcePublishEventsForTwitch      *bool  `json:"force-publish-events-for-twitch,omitempty"`
	ForcePublishEventsForLvs         *bool  `json:"force-publish-events-for-lvs,omitempty"`
	EnablePublishEventsToSec         *bool  `json:"enable-publish-events-to-sec,omitempty"`
	TranscodeHeartbeatInterval       *int32 `json:"transcode-heartbeat-interval,omitempty"`
	StarvationHeartbeatInterval      *int32 `json:"starvation-heartbeat-interval,omitempty"`
}

func StartConfigSDCAgent(ctx context.Context, sess *session.Session, configName string) (*sdcConfigs, error) {
	c := &sdcConfigs{}
	err := startSDCAgent(ctx, sess, c, 30*time.Second, configName)
	if err != nil {
		return nil, err
	}
	return c, nil
}

func (s *sdcConfigs) GetSECProducerConfig() *Configs {
	s.mu.RLock()
	defer s.mu.RUnlock()
	return &Configs{
		EnablePublishEventsToSec:         s.enablePublishEventsToSec,
		EnablePublishStarvationHeartbeat: s.enablePublishStarvationHeartbeat,
		EnablePublishTranscodeHeartbeat:  s.enablePublishTranscodeHeartbeat,
		ForcePublishEventsForTwitch:      s.forcePublishEventsForTwitch,
		ForcePublishEventsForLvs:         s.forcePublishEventsForLvs,
		TranscodeHeartbeatInterval:       s.transcodeHeartbeatInterval,
		StarvationHeartbeatInterval:      s.starvationHeartbeatInterval,
	}
}

func (s *sdcConfigs) GetEnablePublishTranscodeHeartbeat(defaultVal bool) bool {
	s.mu.RLock()
	defer s.mu.RUnlock()
	val := s.enablePublishTranscodeHeartbeat
	if val == nil {
		log.Printf("Unable to get nil enablePublishTranscodeHeartbeat from config")
		return defaultVal
	}
	log.Printf("Got value: %v from sdc for enablePublishTranscodeHeartbeat", *s.enablePublishTranscodeHeartbeat)
	return *s.enablePublishTranscodeHeartbeat
}

func (s *sdcConfigs) GetEnablePublishStarvationHeartbeat(defaultVal bool) bool {
	s.mu.RLock()
	defer s.mu.RUnlock()
	val := s.enablePublishStarvationHeartbeat
	if val == nil {
		log.Printf("Unable to get nil enablePublishStarvationHeartbeat from config")
		return defaultVal
	}
	log.Printf("Got value: %v from sdc for enablePublishStarvationHeartbeat", *s.enablePublishStarvationHeartbeat)
	return *s.enablePublishStarvationHeartbeat
}

func (s *sdcConfigs) GetForcePublishEventsForTwitch(defaultVal bool) bool {
	s.mu.RLock()
	defer s.mu.RUnlock()
	val := s.forcePublishEventsForTwitch
	if val == nil {
		log.Printf("Unable to get nil forcePublishEventsForTwitch from config")
		return defaultVal
	}
	log.Printf("Got value: %v from sdc for forcePublishEventsForTwitch", *s.forcePublishEventsForTwitch)
	return *s.forcePublishEventsForTwitch
}

func (s *sdcConfigs) GetForcePublishEventsForLvs(defaultVal bool) bool {
	s.mu.RLock()
	defer s.mu.RUnlock()
	val := s.forcePublishEventsForLvs
	if val == nil {
		log.Printf("Unable to get nil forcePublishEventsForLvs from config")
		return defaultVal
	}
	log.Printf("Got value: %v from sdc for forcePublishEventsForLvs", *s.forcePublishEventsForLvs)
	return *s.forcePublishEventsForLvs
}

func (s *sdcConfigs) GetEnablePublishEventsToSec(defaultVal bool) bool {
	s.mu.RLock()
	defer s.mu.RUnlock()
	val := s.enablePublishEventsToSec
	if val == nil {
		log.Printf("Unable to get nil enablePublishEventsToSec from config")
		return defaultVal
	}
	log.Printf("Got value: %v from sdc for enablePublishEventsToSec ", *s.enablePublishEventsToSec)
	return *s.enablePublishEventsToSec
}

func (s *sdcConfigs) GetTranscodeHeartbeatInterval(defaultVal int32) int32 {
	s.mu.RLock()
	defer s.mu.RUnlock()
	val := s.transcodeHeartbeatInterval
	if val == nil {
		log.Printf("Unable to get nil transcodeHeartbeatInterval from config")
		return defaultVal
	}
	log.Printf("Got value: %v from sdc for transcodeHeartbeatInterval", *s.transcodeHeartbeatInterval)
	return *s.transcodeHeartbeatInterval
}

func (s *sdcConfigs) GetStarvationHeartbeatInterval(defaultVal int32) int32 {
	s.mu.RLock()
	defer s.mu.RUnlock()
	val := s.starvationHeartbeatInterval
	if val == nil {
		log.Printf("Unable to get nil starvationHeartbeatInterval from config")
		return defaultVal
	}
	log.Printf("Got value: %v from sdc for starvationHeartbeatInterval", *s.starvationHeartbeatInterval)
	return *s.starvationHeartbeatInterval
}

func startSDCAgent(ctx context.Context, sess *session.Session, updater sdcagent.ConfigUpdater, pollInterval time.Duration, configName string) error {
	hostname, err := os.Hostname()
	if err != nil {
		log.Printf("Unable to determine hostname: %q", err)
		hostname = "defaultHostname"
	}

	agent, err := sdcagent.New(&sdcagent.Config{
		Region: *sess.Config.Region,
		OnError: func(err error) {
			log.Printf("SdcAgent %s err=%q\n", configName, err)
		},
		Updater:                updater,
		PollInterval:           pollInterval,
		SdcConfigName:          configName,
		RipService:             "sec",
		SdcConfigSchemaVersion: 2,
		Cloudwatch:             cloudwatch.New(sess),
		HostName:               hostname,
	})
	if err != nil {
		return fmt.Errorf("create sdc agent for config=%q: %w", configName, err)
	}

	go agent.Run(ctx)
	return nil
}
