package StarfruitSECProducer

import (
	"context"

	"code.justin.tv/amzn/StarfruitSECTwirp"

	"github.com/aws/aws-sdk-go/service/sqs/sqsiface"
	"github.com/pkg/errors"
)

type SECProducer interface {
	SendEventToSEC(ctx context.Context, msg *StarfruitSECTwirp.StreamEventData) error
}

// SECSQSProducer is a client to ship data to the SEC.
type SECSQSProducer struct {
	stage      string
	homeRegion string
	sqs        sqsiface.SQSAPI
}

// NewSECProducers creates an SECProducer client which sends data to the SEC.
// It sends it for a particular stage ('prod', 'beta', or 'gamma'), and
// to queues housed within a particular home region ('us-west-2', 'us-east-2').
func NewSECProducer(stage string, homeRegion string, sqs sqsiface.SQSAPI) (SECProducer, error) {

	err := ValidateProducerParams(stage, homeRegion)
	if err != nil {
		return nil, errors.Wrap(err, "SECProducer parameters are invalid")
	}

	return &SECSQSProducer{
		stage:      stage,
		homeRegion: homeRegion,
		sqs:        sqs,
	}, nil
}

// SendEventToSEC sends a single event to SEC
func (p *SECSQSProducer) SendEventToSEC(ctx context.Context, msg *StarfruitSECTwirp.StreamEventData) error {
	homeRegion, err := GetHomeRegionForChannel(msg.GetChannelArn())
	if err != nil {
		return errors.Wrap(err, "unable to determine home region for channel")
	}

	qurl, err := GetQueueUrl(p.stage, homeRegion)
	if err != nil {
		return errors.Wrap(err, "unable to determine destination queue for message")
	}

	batchMsg := &StarfruitSECTwirp.StreamEventDataBatch{
		Data: []*StarfruitSECTwirp.StreamEventData{msg},
	}

	protoProducer := NewProtoSQSProducer(qurl, p.sqs)
	return protoProducer.Send(ctx, batchMsg)
}
