package errorbucket

import (
	"sync"
	"sync/atomic"
)

// Bucket is initialized with a static token count, and will allow up to Count number of tokens
// to be removed from the bucket. Bucket is safe for concurrent use.
type Bucket struct {
	Count uint64

	once   sync.Once
	tokens uint64
}

func (b *Bucket) Allow() bool {
	b.once.Do(func() {
		b.tokens = b.Count
	})

	// Goroutines load the current value of tokens, returning immediately if none are left.
	// Otherwise, it attempts to acquire a token, returning immediately if successful.
	// A goroutine may make several attempts to grab a token, but will always return without
	// blocking once no tokens are left.
	for {
		v := atomic.LoadUint64(&b.tokens)
		if v == 0 {
			return false
		}

		ok := atomic.CompareAndSwapUint64(&b.tokens, v, v-1)
		if ok {
			return true
		}
	}
}
