package timbersaw

import (
	"context"
	"sort"
	"sync"
	"time"

	"code.justin.tv/video/invoker"
)

const updaterFrequency = 1 * time.Second
const refreshGracePeriod = 10 * time.Second

type Cache struct {
	cacheMaxNumOfEntries int
	removeThreshold      time.Duration // how long an entry stays untouched before being removed

	lru *LRUCache

	mu    sync.RWMutex
	items map[CacheKey]*CacheEntry

	refreshMu    sync.Mutex
	refreshKeys  []*RefreshData
	refreshIndex int
}

type CacheKey struct {
	ChannelARN string
	Origin     string
}

type CacheEntry struct {
	sync.RWMutex
	item           []Path
	expiry         time.Time
	lastSet        time.Time
	lastUsed       time.Time
	sentForRefresh time.Time
}

type Path []string

type RefreshData struct {
	Key   *CacheKey
	Entry *CacheEntry
}

func NewCache(maxNumOfEntries int, removeThreshold time.Duration) *Cache {
	return &Cache{
		cacheMaxNumOfEntries: maxNumOfEntries,
		removeThreshold:      removeThreshold,
		lru:                  NewLRU(),
		items:                make(map[CacheKey]*CacheEntry, maxNumOfEntries),
	}
}

// Periodically run tasks
func (c *Cache) Run(ctx context.Context) error {
	i := invoker.New()
	i.Add(c.runCleanup)

	return i.Run(ctx)
}

// Get returns a slice of Paths and boolean value to indicate if it's in the cache.
// Will return a nil value of paths if the key is not in the cache.
func (c *Cache) Get(key CacheKey) ([]Path, bool) {
	c.mu.RLock()
	p, exists := c.items[key]
	c.mu.RUnlock()
	if !exists {
		return nil, false
	}

	c.lru.Add(key)
	p.Lock()
	defer p.Unlock()
	p.lastUsed = time.Now()
	return p.item, true
}

// Set sets the key & value pair in the cache. If the entry is present then it updates it with the new
// value and expiration.
func (c *Cache) Set(key CacheKey, value []Path, newExpiry time.Time) {
	c.mu.RLock()
	p, exists := c.items[key]
	c.mu.RUnlock()

	if exists {
		updateEntry(p, value, newExpiry)
		return
	}

	// another thread calling Set() can mutate the cache and add an entry so
	// recheck that it still doesn't exist
	c.mu.Lock()
	defer c.mu.Unlock()
	p, exists = c.items[key]
	if exists {
		updateEntry(p, value, newExpiry)
		return
	}

	// enforce LRU policy if cache is full
	for c.lru.Len() >= c.cacheMaxNumOfEntries {
		oldestItem := c.lru.GetOldest()
		delete(c.items, oldestItem)
		c.lru.Remove(oldestItem)
	}

	c.items[key] = &CacheEntry{
		item:           value,
		expiry:         newExpiry,
		lastSet:        time.Now(),
		lastUsed:       time.Now(), // because it wasn't in the map before, we'll give it the benefit of the doubt on whether its used or not
		sentForRefresh: time.Time{},
	}
	c.lru.Add(key)
}

// NextRefreshKeys returns a list of the next `count` cache items that need to be refreshed
// If there are less than `count` items left it loops the list to ensure it returns `count` items
// It uses data set by `cleanup` and therefore can be up to `updaterFrequency` out of date
// It also returns the max duration since last set for tracking the refresh interval
func (c *Cache) NextRefreshKeys(count int) ([]CacheKey, time.Duration) {
	data := make([]*RefreshData, 0, count)

	c.refreshMu.Lock()
	for len(c.refreshKeys) > 0 && len(data) < count {
		nextIndex := c.refreshIndex + count - len(data)
		if nextIndex > len(c.refreshKeys) {
			nextIndex = len(c.refreshKeys)
		}

		data = append(data, c.refreshKeys[c.refreshIndex:nextIndex]...)
		c.refreshIndex = nextIndex
		if c.refreshIndex >= len(c.refreshKeys) {
			c.refreshIndex = 0
		}
	}
	c.refreshMu.Unlock()

	now := time.Now()
	refreshAge := time.Duration(0)
	for _, d := range data {
		d.Entry.Lock()
		d.Entry.sentForRefresh = now
		age := now.Sub(d.Entry.lastSet)
		if age > refreshAge {
			refreshAge = age
		}
		d.Entry.Unlock()
	}

	keys := make([]CacheKey, 0, len(data))
	for _, d := range data {
		keys = append(keys, *d.Key)
	}

	return keys, refreshAge
}

func updateEntry(p *CacheEntry, value []Path, newExpiry time.Time) {
	p.Lock()
	defer p.Unlock()
	p.expiry = newExpiry
	p.item = value
	p.lastSet = time.Now()
	p.sentForRefresh = time.Time{}
}

// Garbage collect entries that haven't been used and build ordered list for refreshing
func (c *Cache) cleanup() {
	// Build a copy of all items in the cache
	c.mu.RLock()
	data := make([]*RefreshData, 0, len(c.items))
	for key, entry := range c.items {
		key := key
		data = append(data, &RefreshData{
			Key:   &key,
			Entry: entry,
		})
	}
	c.mu.RUnlock()

	// Filter out items in our copy that are recently requested to refresh or should be removed entirely
	now := time.Now()
	toRemove := []*CacheKey{}
	for idx, d := range data {
		d.Entry.RLock()

		// Filter entries that have recently been requested
		if d.Entry.sentForRefresh.Add(refreshGracePeriod).After(now) {
			data[idx] = nil
		}

		// Filter & remove entries that have expired
		if now.Sub(d.Entry.lastUsed) > c.removeThreshold {
			data[idx] = nil
			toRemove = append(toRemove, d.Key)
		}

		d.Entry.RUnlock()
	}

	// Actually remove the entries
	c.mu.Lock()
	for _, k := range toRemove {
		delete(c.items, *k)
		c.lru.Remove(*k)
	}
	c.mu.Unlock()

	// Sort our filtered copy, putting nils at the end
	// Note: Our caller also relies on this ordering.
	sort.Slice(data, func(i, j int) bool {
		a, b := data[i], data[j]
		if a == nil {
			return false
		}
		if b == nil {
			return true
		}
		return a.Entry.expiry.Before(b.Entry.expiry)
	})

	// Remove the nils at the end
	for len(data) > 0 && data[len(data)-1] == nil {
		data = data[:len(data)-1]
	}

	// Save the filtered copy as the new list of refreshable keys
	c.refreshMu.Lock()
	c.refreshKeys = data
	c.refreshIndex = 0
	c.refreshMu.Unlock()
}

func (c *Cache) runCleanup(ctx context.Context) error {
	ticker := time.NewTicker(updaterFrequency)
	defer ticker.Stop()

	for {
		select {
		case <-ctx.Done():
			return ctx.Err()
		case <-ticker.C:
			c.cleanup()
		}
	}
}
