package timbersaw

import (
	"container/list"
	"sync"
)

// LRUCache is a threadsafe wrapper around container/list
type LRUCache struct {
	mu sync.Mutex
	h  map[CacheKey]*list.Element
	l  *list.List
}

func NewLRU() *LRUCache {
	return &LRUCache{
		h: make(map[CacheKey]*list.Element),
		l: list.New(),
	}
}

// Add brings the key to the front of the list if present.
func (c *LRUCache) Add(key CacheKey) {
	c.mu.Lock()
	defer c.mu.Unlock()
	elem, ok := c.h[key]
	if ok {
		c.l.MoveToFront(elem)
		return
	}
	newElem := c.l.PushFront(key)
	c.h[key] = newElem
}

// Remove returns true if key is present and removed.
func (c *LRUCache) Remove(key CacheKey) bool {
	c.mu.Lock()
	defer c.mu.Unlock()
	elem, ok := c.h[key]
	if ok {
		delete(c.h, elem.Value.(CacheKey))
		c.l.Remove(elem)
		return true
	}
	return false
}

func (c *LRUCache) GetOldest() CacheKey {
	c.mu.Lock()
	defer c.mu.Unlock()
	if c.l.Len() == 0 {
		return CacheKey{}
	}
	return c.l.Back().Value.(CacheKey)
}

func (c *LRUCache) Len() int {
	c.mu.Lock()
	defer c.mu.Unlock()
	return c.l.Len()
}
