package vault

import (
	"fmt"
	"sync"
	"sync/atomic"

	"code.justin.tv/amzn/StarfruitVault/key"
	token "code.justin.tv/amzn/StarfruitVaultToken"

	"github.com/gogo/protobuf/proto"
)

type (
	TokenVersion = token.Token_Version
	Token        = token.Token
)

const (
	TokenV1 = token.Token_V1
	TokenV2 = token.Token_V2
)

var DefaultTokens = []TokenVersion{TokenV1, TokenV2}

type DecodeVaultConfig struct {
	// V1SecretAlias (Optional) specifies the Producer ID used V1 schema.
	V1SecretAlias string

	// SupportedVersions: Defaults to all tokens
	SupportedVersions []TokenVersion

	IgnoreDeactivated bool
}

type DecodeVault interface {
	Decode(input []byte) (secretAlias string, plaintext []byte, err error)
}

type decodeVault struct {
	DecodeVaultConfig

	// bitmap of supported versions
	supportedVersions uint64

	keys       atomic.Value
	updateLock sync.Mutex
}

func NewDecodeVault(config *DecodeVaultConfig) (*decodeVault, error) {
	if config == nil {
		return nil, fmt.Errorf("config cannot be nil")
	}

	if len(config.SupportedVersions) == 0 {
		config.SupportedVersions = DefaultTokens
	}

	var supportedVersions uint64
	for _, version := range config.SupportedVersions {
		supportedVersions = supportedVersions | (1 << version)
	}
	return &decodeVault{DecodeVaultConfig: *config, supportedVersions: supportedVersions}, nil
}

func (v *decodeVault) getKeys() map[string]map[uint32]*key.Key {
	value := v.keys.Load()
	if value == nil {
		return nil
	}
	return value.(map[string]map[uint32]*key.Key)
}

func (v *decodeVault) versionSupported(version TokenVersion) bool {
	return (v.supportedVersions & (1 << version)) != 0
}

func (v *decodeVault) OnKeys(secretAlias string, set *key.KeySet) {
	keys := v.activeKeys(set)

	v.updateLock.Lock()
	defer v.updateLock.Unlock()

	producers := v.getKeys()
	newProducers := make(map[string]map[uint32]*key.Key, len(producers))
	for k, v := range producers {
		newProducers[k] = v
	}
	newProducers[secretAlias] = keys
	v.keys.Store(newProducers)
}

var _ DecodeVault = (*decodeVault)(nil)

func (v *decodeVault) Decode(input []byte) (string, []byte, error) {
	var t token.Token
	err := proto.Unmarshal(input, &t)
	if err != nil {
		return "", nil, err
	}

	if !v.versionSupported(t.Version) {
		return "", nil, fmt.Errorf("token version not supported")
	}

	switch t.Version {
	case TokenV1:
		return v.decodeV1(&t)
	case TokenV2:
		return v.decodeV2(&t)
	default:
		return "", nil, fmt.Errorf("unknown token version")
	}
}

func (v *decodeVault) decodeV1(t *token.Token) (string, []byte, error) {
	if v.V1SecretAlias == "" {
		return "", nil, fmt.Errorf("token version v1 not supported")
	}
	t.SecretAlias = v.V1SecretAlias
	t.CipherText = append(t.CipherText, t.Signature...)
	return v.decodeV2(t)
}

func (v *decodeVault) decodeV2(t *token.Token) (string, []byte, error) {
	allKeys := v.getKeys()

	keysForAlias, ok := allKeys[t.SecretAlias]
	if !ok {
		return "", nil, fmt.Errorf("producer id not found: %s", t.SecretAlias)
	}

	key, ok := keysForAlias[t.KeyEpoch]
	if !ok {
		return "", nil, fmt.Errorf("key epoch not found %s:%d", t.SecretAlias, t.KeyEpoch)
	}

	ok, err := key.Verify(t.CipherText, t.Signature)
	if err != nil || !ok {
		return "", nil, fmt.Errorf("unable to verify data for producer(%s:%d): %w", t.SecretAlias, t.KeyEpoch, err)
	}

	decoded, err := key.Decrypt(t.Nonce, t.CipherText)
	if err != nil {
		return "", nil, fmt.Errorf("unable to decode data for producer(%s:%d): %w", t.SecretAlias, t.KeyEpoch, err)
	}

	return t.SecretAlias, decoded, nil
}

func (v *decodeVault) activeKeys(set *key.KeySet) map[uint32]*key.Key {
	keys := make(map[uint32]*key.Key, len(set.Keys))
	for _, key := range set.Keys {
		if v.IgnoreDeactivated || !key.Deactivated {
			keys[key.KeyEpoch] = key
		}
	}
	return keys
}
