package vault_test

import (
	"crypto/rand"
	"fmt"
	"testing"

	vault "code.justin.tv/amzn/StarfruitVault"

	"code.justin.tv/amzn/StarfruitVault/key"
)

func BenchmarkDecodeVault(b *testing.B) {
	benchmarks := []struct {
		EncryptionKeyType key.KeyType
		SignatureKeyType  key.KeyType
	}{
		{key.KeyTypeSharedAESGCM, key.KeyTypeNoop},
		{key.KeyTypeSharedAESGCM, key.KeyTypePrivateEd25519},
		{key.KeyTypeSharedSecretBox, key.KeyTypePrivateEd25519},
		{key.KeyTypeSharedSecretBox, key.KeyTypePrivateEd25519},
	}
	sizes := []int{64, 128, 256, 512, 1024, 2048, 4096}
	buf := make([]byte, 0, 4096)
	_, err := rand.Read(buf)
	if err != nil {
		b.Fatal(err)
	}

	for _, bm := range benchmarks {
		k, err := key.GenerateKey(&key.GenerateOptions{
			EncryptionKeyType: bm.EncryptionKeyType,
			SignatureKeyType:  bm.SignatureKeyType,
		})
		if err != nil {
			b.Fatal(err)
		}

		enc, err := vault.NewEncodeVault(&vault.EncodeVaultConfig{})
		if err != nil {
			b.Fatal(err)
		}

		dec, err := vault.NewDecodeVault(&vault.DecodeVaultConfig{})
		if err != nil {
			b.Fatal(err)
		}
		dk, err := k.ConsumerKey()
		if err != nil {
			b.Fatal(err)
		}

		enc.OnKeys("test", &key.KeySet{Keys: []*key.Key{k}})
		dec.OnKeys("test", &key.KeySet{Keys: []*key.Key{dk}})

		for _, sz := range sizes {
			buf, err := enc.Encode("test", buf[:sz])
			if err != nil {
				b.Fatal(err)
			}

			b.Run(fmt.Sprintf("%s_%s_%d", dk.EncryptionKey.KeyType(), dk.SignatureKey.KeyType(), sz), func(b *testing.B) {
				b.RunParallel(func(pb *testing.PB) {
					for pb.Next() {
						_, _, _ = dec.Decode(buf)
					}
				})
			})
		}
	}
}
