package vault

import (
	"fmt"
	"sort"
	"sync"
	"sync/atomic"

	"golang.org/x/crypto/sha3"

	"code.justin.tv/amzn/StarfruitVault/key"

	"github.com/gogo/protobuf/proto"
)

type EncodeVault interface {
	Encode(consumer string, input []byte) ([]byte, error)
}

type encodeVault struct {
	keys       atomic.Value
	updateLock sync.Mutex
}

var _ EncodeVault = (*encodeVault)(nil)

type EncodeVaultConfig struct{}

func NewEncodeVault(config *EncodeVaultConfig) (*encodeVault, error) {
	if config == nil {
		return nil, fmt.Errorf("config cannot be nil")
	}
	return &encodeVault{}, nil
}

func (v *encodeVault) OnKeys(secretAlias string, set *key.KeySet) {
	newKey, _ := ActiveKey(set)
	if newKey == nil {
		return
	}

	v.updateLock.Lock()
	defer v.updateLock.Unlock()

	keys := v.getKeys()
	newKeys := make(map[string]*key.Key, len(keys))
	for k, v := range keys {
		newKeys[k] = v
	}
	newKeys[secretAlias] = newKey
	v.keys.Store(newKeys)
}

// use the second most recent non deactivated key
func ActiveKey(set *key.KeySet) (*key.Key, error) {
	if len(set.Keys) == 0 {
		return nil, fmt.Errorf("no keys")
	}

	sort.Slice(set.Keys, func(i int, j int) bool {
		if set.Keys[j] == nil {
			return true
		}
		if set.Keys[i] == nil {
			return false
		}

		// deactivated keys should go to the front
		if !set.Keys[i].Deactivated && set.Keys[j].Deactivated {
			return false
		}
		return set.Keys[i].GeneratedAt.Before(set.Keys[j].GeneratedAt)
	})

	activeCount := 0
	var active *key.Key
	// iterate in reverse
	for i := len(set.Keys) - 1; i >= 0; i-- {
		k := set.Keys[i]
		if k == nil {
			continue
		}
		if !k.Deactivated {
			active = k
			activeCount++
		}

		if activeCount == 2 {
			break
		}
	}
	if active == nil {
		return nil, fmt.Errorf("no active key")
	}
	return active, nil
}

func (v *encodeVault) getKeys() map[string]*key.Key {
	c := v.keys.Load()
	if c == nil {
		return nil
	}
	return c.(map[string]*key.Key)
}

func (v *encodeVault) Encode(secretAlias string, input []byte) ([]byte, error) {
	var err error

	keys := v.getKeys()
	key, ok := keys[secretAlias]
	if !ok {
		return nil, fmt.Errorf("secret alias not found: %s", secretAlias)
	}

	t := &Token{
		Version:     TokenV2,
		Nonce:       make([]byte, key.NonceSize()),
		SecretAlias: secretAlias,
		KeyEpoch:    key.GetKeyEpoch(),
	}

	// Iv has to be globally unique and we are using a hash of the message to
	// make the resulting payload string deterministic
	sha3.ShakeSum128(t.Nonce, input)

	t.CipherText, err = key.Encrypt(t.Nonce, input)
	if err != nil {
		return nil, fmt.Errorf("could not encrypt payload: %w", err)
	}
	t.Signature, err = key.Sign(t.CipherText)
	if err != nil {
		return nil, fmt.Errorf("could not sign payload: %w", err)
	}

	return proto.Marshal(t)
}
