package vault_test

import (
	"crypto/rand"
	"fmt"
	"testing"
	"time"

	vault "code.justin.tv/amzn/StarfruitVault"

	"code.justin.tv/amzn/StarfruitVault/key"
)

func TestKeySort(t *testing.T) {
	var keys key.KeySet
	var expectedActive *key.Key

	for i := uint32(0); i < 30; i++ {
		k, err := key.GenerateKey(&key.GenerateOptions{
			EncryptionKeyType: key.KeyTypeSharedAESGCM,
			SignatureKeyType:  key.KeyTypePrivateEd25519,
			KeyEpoch:          i,
			GeneratedAt:       time.Now().Add(-time.Duration(30-i) * time.Hour),
		})
		if err != nil {
			t.Fatal(err)
		}

		if i < 10 {
			k.Deactivated = true
		}

		if i == 28 {
			expectedActive = k
		}
		keys.Keys = append(keys.Keys, k)
	}

	k, err := vault.ActiveKey(&keys)
	if err != nil {
		t.Fatal(err)
	}

	if k != expectedActive {
		t.Fatalf("expected key %d != %d", expectedActive.KeyEpoch, k.KeyEpoch)
	}
}

func BenchmarkEncodeVault(b *testing.B) {
	benchmarks := []struct {
		EncryptionKeyType key.KeyType
		SignatureKeyType  key.KeyType
	}{
		{key.KeyTypeSharedAESGCM, key.KeyTypeNoop},
		{key.KeyTypeSharedAESGCM, key.KeyTypePrivateEd25519},
		{key.KeyTypeSharedSecretBox, key.KeyTypePrivateEd25519},
		{key.KeyTypeSharedSecretBox, key.KeyTypePrivateEd25519},
	}
	sizes := []int{64, 128, 256, 512, 1024, 2048, 4096}
	buf := make([]byte, 0, 4096)
	_, err := rand.Read(buf)
	if err != nil {
		b.Fatal(err)
	}

	for _, bm := range benchmarks {
		k, err := key.GenerateKey(&key.GenerateOptions{
			EncryptionKeyType: bm.EncryptionKeyType,
			SignatureKeyType:  bm.SignatureKeyType,
		})
		if err != nil {
			b.Fatal(err)
		}

		v, err := vault.NewEncodeVault(&vault.EncodeVaultConfig{})
		if err != nil {
			b.Fatal(err)
		}

		v.OnKeys("test", &key.KeySet{Keys: []*key.Key{k}})
		for _, sz := range sizes {
			b.Run(fmt.Sprintf("%s_%s_%d", bm.EncryptionKeyType, bm.SignatureKeyType, sz), func(b *testing.B) {
				b.RunParallel(func(pb *testing.PB) {
					for pb.Next() {
						_, _ = v.Encode("test", buf[:sz])
					}
				})
			})
		}
	}
}
