package key

import (
	"crypto/aes"
	"crypto/cipher"
	"crypto/rand"
	"fmt"
)

const (
	// 256-bit AES by default
	aesBlockSize = 32
)

type aesKey struct {
	key []byte
	gcm cipher.AEAD
}

func (k *aesKey) KeyType() KeyType {
	return KeyTypeSharedAESGCM
}

func (k *aesKey) Key() []byte {
	return k.key
}

var _ EncryptionKey = (*aesKey)(nil)

func GenerateAESGCMKey() (*aesKey, error) {
	var sharedSecret [aesBlockSize]byte
	n, err := rand.Read(sharedSecret[:])
	if err != nil {
		return nil, fmt.Errorf("unable to generate encryption Key: %w", err)
	}
	if n != len(sharedSecret) {
		return nil, fmt.Errorf("unabled to read 32 bytes from crypto/rand")
	}
	return newAESGCMKey(sharedSecret[:])
}

func newAESGCMKey(key []byte) (*aesKey, error) {
	block, err := aes.NewCipher(key)
	if err != nil {
		return nil, fmt.Errorf("unable to create aes block from serialized Key: %w", err)
	}
	gcm, err := cipher.NewGCM(block)
	if err != nil {
		return nil, fmt.Errorf("unable to create gcm from serialized Key: %w", err)
	}

	return &aesKey{key, gcm}, nil
}

func (k *aesKey) NonceSize() int {
	return k.gcm.NonceSize()
}

func (k *aesKey) Encrypt(nonce []byte, input []byte) (ciphertext []byte, err error) {
	sealedMessage := k.gcm.Seal(nil, nonce, input, nil)
	return sealedMessage, nil
}

func (k *aesKey) Decrypt(nonce []byte, ciphertext []byte) ([]byte, error) {
	nonceSize := k.gcm.NonceSize()
	if len(nonce) != nonceSize {
		return nil, fmt.Errorf("nonce is of incorrect size, expected(%d) != actual(%d)", nonceSize, len(nonce))
	}
	return k.gcm.Open(nil, nonce, ciphertext, nil)
}
