package key

import (
	"crypto/ed25519"
	"fmt"

	"golang.org/x/crypto/nacl/sign"
)

type ed25519PublicKey struct {
	key []byte
}

func (k *ed25519PublicKey) KeyType() KeyType {
	return KeyTypePublicEd25519
}

func (k *ed25519PublicKey) Key() []byte {
	return k.key
}

var _ SignatureKey = (*ed25519PublicKey)(nil)

func newEd25519PublicKey(key []byte) (*ed25519PublicKey, error) {
	return &ed25519PublicKey{key}, nil
}

func (k *ed25519PublicKey) Sign(_ []byte) ([]byte, error) {
	return nil, fmt.Errorf("cannot sign with the given Key type (%s)", KeyTypePublicEd25519)
}

func (k *ed25519PublicKey) Verify(input, sig []byte) (bool, error) {
	return ed25519.Verify(k.key, input, sig), nil
}

func (k *ed25519PublicKey) Public() (SignatureKey, error) {
	return k, nil
}

type ed25519PrivateKey struct {
	key []byte
}

func (k *ed25519PrivateKey) KeyType() KeyType {
	return KeyTypePrivateEd25519
}

func (k *ed25519PrivateKey) Key() []byte {
	return k.key
}

var _ SignatureKey = (*ed25519PrivateKey)(nil)

func GenerateEd25519PrivateKey() (*ed25519PrivateKey, error) {
	_, private, err := sign.GenerateKey(nil)
	if err != nil {
		return nil, fmt.Errorf("unable to generate ed25519 private Key: %w", err)
	}
	return &ed25519PrivateKey{key: private[:]}, nil
}

func newEd25519PrivateKey(key []byte) (*ed25519PrivateKey, error) {
	return &ed25519PrivateKey{key}, nil
}

func (k *ed25519PrivateKey) Public() (SignatureKey, error) {
	key, ok := ed25519.PrivateKey(k.key).Public().(ed25519.PublicKey)
	if !ok {
		return nil, fmt.Errorf("unable to convert private Key to public Key")
	}
	return &ed25519PublicKey{key}, nil
}

func (k *ed25519PrivateKey) Sign(input []byte) ([]byte, error) {
	return ed25519.Sign(k.key, input), nil
}

func (k *ed25519PrivateKey) Verify(input, sig []byte) (bool, error) {
	public, err := k.Public()
	if err != nil {
		return false, err
	}

	return public.Verify(input, sig)
}
